"use client";
import React, { useState, useEffect } from "react";
import axios from "axios";
import {
  Grid,
  Typography,
  Box,
  Card,
  CardContent,
  Button,
  Chip,
  Divider,
  Paper,
  Stack,
  Table,
  TableBody,
  TableCell,
  TableContainer,
  TableHead,
  TableRow,
  Pagination,
  useMediaQuery,
  Theme,
} from "@mui/material";
import { DatePicker } from "@mui/x-date-pickers";
import { LocalizationProvider } from "@mui/x-date-pickers/LocalizationProvider";
import { AdapterLuxon } from "@mui/x-date-pickers/AdapterLuxon";
import LoadingComponent from "@/app/components/loadingcomponent";
import { DateTime } from "luxon";
import {
  PieChart,
  Pie,
  Cell,
  Tooltip,
  BarChart,
  Bar,
  XAxis,
  YAxis,
  CartesianGrid,
  Legend,
  LineChart,
  Line,
  ResponsiveContainer,
} from "recharts";

type Transaction = {
  _id: string;
  type: string;
  amount: number;
  status: string;
  createdAt: string;
  userEmail: string;
  userId?: string;
  network?: string;
  planType?: string;
};

const DashboardOverview: React.FC = () => {
  const isMobile = useMediaQuery((theme: Theme) =>
    theme.breakpoints.down("sm")
  );
  const isTablet = useMediaQuery((theme: Theme) =>
    theme.breakpoints.down("md")
  );

  const [startDate, setStartDate] = useState<DateTime | null>(
    DateTime.now().minus({ days: 30 })
  );
  const [endDate, setEndDate] = useState<DateTime | null>(DateTime.now());
  const [loading, setLoading] = useState(false);
  const [activeTab, setActiveTab] = useState(0);
  const [currentPage, setCurrentPage] = useState(1);
  const [transactionsPerPage] = useState(5);

  const [data, setData] = useState({
    metrics: {
      totalTransactions: 0,
      totalRevenue: 0,
      totalUsers: 0,
      totalWalletBalance: 0,
      profit: 0,
      activeUsers: 0,
      newUsers: 0,
      dataTransactions: 0,
      airtimeTransactions: 0,
      cableTransactions: 0,
      billTransactions: 0,
    },
    transactionStats: {
      pending: 0,
      successful: 0,
      failed: 0,
    },
    revenueByService: {
      data: 0,
      airtime: 0,
      cable: 0,
      bills: 0,
    },
    dailyTransactions: [] as any[],
    recentTransactions: [] as Transaction[],
    totalTransactionCount: 0,
  });

  const fetchData = async () => {
    if (startDate && endDate) {
      setLoading(true);
      try {
        const response = await axios.get("/api/adminapi/dashboard", {
          params: {
            startDate: startDate.toISO(),
            endDate: endDate.toISO(),
            page: currentPage,
            limit: transactionsPerPage,
          },
        });
        setData(response.data);
      } catch (error) {
        console.error("Error fetching data:", error);
      } finally {
        setLoading(false);
      }
    }
  };

  useEffect(() => {
    fetchData();
  }, [startDate, endDate, currentPage]);

  const handlePageChange = (
    event: React.ChangeEvent<unknown>,
    value: number
  ) => {
    setCurrentPage(value);
  };

  // Chart data preparation
  const serviceRevenueData = [
    { name: "Data", value: data.revenueByService.data },
    { name: "Airtime", value: data.revenueByService.airtime },
    { name: "Cable", value: data.revenueByService.cable },
    { name: "Bills", value: data.revenueByService.bills },
  ];

  const transactionStatusData = [
    { name: "Successful", value: data.transactionStats.successful },
    { name: "Pending", value: data.transactionStats.pending },
    { name: "Failed", value: data.transactionStats.failed },
  ];

  const COLORS = ["#0088FE", "#00C49F", "#FFBB28", "#FF8042"];

  const formatCurrency = (amount: number) => {
    return new Intl.NumberFormat("en-NG", {
      style: "currency",
      currency: "NGN",
    }).format(amount);
  };

  const formatDate = (dateString: string) => {
    return DateTime.fromISO(dateString).toFormat("MMM dd, yyyy HH:mm");
  };

  const getTransactionTypeLabel = (type: string) => {
    const typeMap: Record<string, string> = {
      DataTransaction: "Data",
      AirtimeTransaction: "Airtime",
      CableSubscription: "Cable",
      BillPayment: "Bill",
      ManualFunding: "Funding",
      AutomaticFunding: "Auto-Funding",
    };
    return typeMap[type] || type;
  };

  return (
    <Box sx={{ p: isMobile ? 1 : 3 }}>
      <Stack
        direction={isMobile ? "column" : "row"}
        justifyContent="space-between"
        alignItems="center"
        mb={3}
        spacing={2}
      >
        <Typography
          variant="h4"
          fontWeight="bold"
          sx={{ fontSize: isMobile ? "1.5rem" : "2rem" }}
        >
          Admin Dashboard
        </Typography>
        <Chip
          label={DateTime.now().toFormat("MMMM dd, yyyy")}
          color="primary"
          variant="outlined"
          size={isMobile ? "small" : "medium"}
        />
      </Stack>

      {/* Date Range Filter */}
      <Paper sx={{ p: isMobile ? 1 : 3, mb: 3 }}>
        <Typography
          variant="h6"
          gutterBottom
          sx={{ fontSize: isMobile ? "1.1rem" : "1.25rem" }}
        >
          Date Range Filter
        </Typography>
        <Divider sx={{ mb: 2 }} />
        <Stack
          direction={isMobile ? "column" : "row"}
          spacing={2}
          alignItems="center"
        >
          <LocalizationProvider dateAdapter={AdapterLuxon}>
            <DatePicker
              label="Start Date"
              value={startDate}
              onChange={(newValue) => setStartDate(newValue)}
              slotProps={{ textField: { size: "small", fullWidth: isMobile } }}
            />
            <DatePicker
              label="End Date"
              value={endDate}
              onChange={(newValue) => setEndDate(newValue)}
              slotProps={{ textField: { size: "small", fullWidth: isMobile } }}
            />
          </LocalizationProvider>
          <Button
            variant="contained"
            onClick={fetchData}
            disabled={!startDate || !endDate}
            sx={{ height: 40, width: isMobile ? "100%" : "auto" }}
          >
            Apply Filter
          </Button>
          <Button
            variant="outlined"
            onClick={() => {
              setStartDate(DateTime.now().minus({ days: 30 }));
              setEndDate(DateTime.now());
            }}
            sx={{ height: 40, width: isMobile ? "100%" : "auto" }}
          >
            Last 30 Days
          </Button>
        </Stack>
      </Paper>

      {loading ? (
        <LoadingComponent />
      ) : (
        <>
          {/* Summary Metrics */}
          <Grid container spacing={isMobile ? 1 : 3} mb={3}>
            {[
              {
                title: "Total Revenue",
                value: formatCurrency(data.metrics.totalRevenue),
                subtext: `${startDate?.toFormat(
                  "MMM dd"
                )} - ${endDate?.toFormat("MMM dd, yyyy")}`,
                color: "primary.main",
              },
              {
                title: "Total Transactions",
                value: data.metrics.totalTransactions.toLocaleString(),
                subtext: (
                  <Stack direction="row" spacing={1} mt={1}>
                    <Chip
                      size="small"
                      label={`${data.transactionStats.successful} success`}
                      color="success"
                    />
                    <Chip
                      size="small"
                      label={`${data.transactionStats.pending} pending`}
                      color="warning"
                    />
                  </Stack>
                ),
                color: "secondary.main",
              },
              {
                title: "Total Users",
                value: data.metrics.totalUsers.toLocaleString(),
                subtext: (
                  <Stack direction="row" spacing={1} mt={1}>
                    <Chip
                      size="small"
                      label={`${data.metrics.newUsers} new`}
                      color="info"
                    />
                    <Chip
                      size="small"
                      label={`${data.metrics.activeUsers} active`}
                      color="success"
                    />
                  </Stack>
                ),
                color: "info.main",
              },
              {
                title: "Wallet Balances",
                value: formatCurrency(data.metrics.totalWalletBalance),
                subtext: "Total user balances",
                color: "success.main",
              },
            ].map((metric, index) => (
              <Grid size={{ xs: 12, sm: 6, md: 3 }} key={index}>
                <Card sx={{ height: "100%", minHeight: isMobile ? 120 : 160 }}>
                  <CardContent>
                    <Typography variant="subtitle2" color="text.secondary">
                      {metric.title}
                    </Typography>
                    <Typography
                      variant="h4"
                      fontWeight="bold"
                      color={metric.color}
                      sx={{ fontSize: isMobile ? "1.5rem" : "2rem" }}
                    >
                      {metric.value}
                    </Typography>
                    {typeof metric.subtext === "string" ? (
                      <Typography variant="caption" color="text.secondary">
                        {metric.subtext}
                      </Typography>
                    ) : (
                      metric.subtext
                    )}
                  </CardContent>
                </Card>
              </Grid>
            ))}
          </Grid>

          {/* Charts Section */}
          <Grid container spacing={isMobile ? 1 : 3} mb={3}>
            <Grid size={{ xs: 12, md: 6 }}>
              <Card>
                <CardContent>
                  <Typography variant="h6" gutterBottom>
                    Revenue by Service
                  </Typography>
                  <Box sx={{ height: 300 }}>
                    <ResponsiveContainer width="100%" height="100%">
                      <PieChart>
                        <Pie
                          data={serviceRevenueData}
                          cx="50%"
                          cy="50%"
                          labelLine={false}
                          outerRadius={80}
                          fill="#8884d8"
                          dataKey="value"
                          label={({ name, percent }) =>
                            `${name}: ${(percent * 100).toFixed(0)}%`
                          }
                        >
                          {serviceRevenueData.map((entry, index) => (
                            <Cell
                              key={`cell-${index}`}
                              fill={COLORS[index % COLORS.length]}
                            />
                          ))}
                        </Pie>
                        <Tooltip
                          formatter={(value) => [
                            formatCurrency(Number(value)),
                            "Revenue",
                          ]}
                        />
                      </PieChart>
                    </ResponsiveContainer>
                  </Box>
                </CardContent>
              </Card>
            </Grid>
            <Grid size={{ xs: 12, md: 6 }}>
              <Card>
                <CardContent>
                  <Typography variant="h6" gutterBottom>
                    Transaction Status
                  </Typography>
                  <Box sx={{ height: 300 }}>
                    <ResponsiveContainer width="100%" height="100%">
                      <PieChart>
                        <Pie
                          data={transactionStatusData}
                          cx="50%"
                          cy="50%"
                          labelLine={false}
                          outerRadius={80}
                          fill="#8884d8"
                          dataKey="value"
                          label={({ name, percent }) =>
                            `${name}: ${(percent * 100).toFixed(0)}%`
                          }
                        >
                          {transactionStatusData.map((entry, index) => (
                            <Cell
                              key={`cell-${index}`}
                              fill={
                                index === 0
                                  ? "#4CAF50"
                                  : index === 1
                                  ? "#FFC107"
                                  : "#F44336"
                              }
                            />
                          ))}
                        </Pie>
                        <Tooltip
                          formatter={(value) => [
                            `${value} transactions`,
                            "Count",
                          ]}
                        />
                      </PieChart>
                    </ResponsiveContainer>
                  </Box>
                </CardContent>
              </Card>
            </Grid>
          </Grid>

          {/* Transaction Trends Chart */}
          <Card sx={{ mb: 3 }}>
            <CardContent>
              <Typography variant="h6" gutterBottom>
                Transaction Trends
              </Typography>
              <Box sx={{ height: 300 }}>
                <ResponsiveContainer width="100%" height="100%">
                  <LineChart data={data.dailyTransactions}>
                    <CartesianGrid strokeDasharray="3 3" />
                    <XAxis dataKey="date" />
                    <YAxis />
                    <Tooltip
                      formatter={(value) => [`${value} transactions`, "Count"]}
                    />
                    <Legend />
                    <Line
                      type="monotone"
                      dataKey="count"
                      stroke="#1976d2"
                      strokeWidth={2}
                      activeDot={{ r: 8 }}
                      name="Transactions"
                    />
                  </LineChart>
                </ResponsiveContainer>
              </Box>
            </CardContent>
          </Card>

          {/* Recent Transactions */}
          <Card>
            <CardContent>
              <Typography variant="h6" gutterBottom>
                Recent Transactions
              </Typography>
              <TableContainer component={Paper} sx={{ mb: 2 }}>
                <Table size={isMobile ? "small" : "medium"}>
                  <TableHead>
                    <TableRow>
                      <TableCell>Type</TableCell>
                      {!isMobile && <TableCell>User</TableCell>}
                      <TableCell>Amount</TableCell>
                      <TableCell>Status</TableCell>
                      {!isMobile && <TableCell>Date</TableCell>}
                    </TableRow>
                  </TableHead>
                  <TableBody>
                    {data.recentTransactions.map((tx) => (
                      <TableRow key={tx._id}>
                        <TableCell>
                          {getTransactionTypeLabel(tx.type)}
                        </TableCell>
                        {!isMobile && (
                          <TableCell>
                            {tx.userEmail
                              ? tx.userEmail.substring(0, 10) + "..."
                              : "System"}
                          </TableCell>
                        )}
                        <TableCell>
                          {tx.amount ? formatCurrency(tx.amount) : "-"}
                        </TableCell>
                        <TableCell>
                          <Chip
                            label={tx.status}
                            size="small"
                            color={
                              tx.status === "Successful"
                                ? "success"
                                : tx.status === "Pending"
                                ? "warning"
                                : "error"
                            }
                          />
                        </TableCell>
                        {!isMobile && (
                          <TableCell>{formatDate(tx.createdAt)}</TableCell>
                        )}
                      </TableRow>
                    ))}
                  </TableBody>
                </Table>
              </TableContainer>
              <Box display="flex" justifyContent="center">
                <Pagination
                  count={Math.ceil(
                    data.totalTransactionCount / transactionsPerPage
                  )}
                  page={currentPage}
                  onChange={handlePageChange}
                  color="primary"
                  size={isMobile ? "small" : "medium"}
                />
              </Box>
            </CardContent>
          </Card>
        </>
      )}
    </Box>
  );
};

export default DashboardOverview;
