// app/api/admin/dashboard/route.ts
import { NextRequest, NextResponse } from "next/server";
import { DateTime } from "luxon";
import Transaction from "@/models/transactions";
import { User } from "@/models/user";
import dbConnect from "@/lib/connectdb";
import { authOptions } from "@/auth";
import { getServerSession } from "next-auth";

export const GET = async (req: NextRequest) => {
  try {
    const session = await getServerSession(authOptions);
    if (!session || session.user.role !== "admin") {
      return NextResponse.json({ error: "Unauthorized" }, { status: 401 });
    }
    await dbConnect();

    // Parse query parameters
    const { searchParams } = new URL(req.url);
    const startDate = searchParams.get("startDate");
    const endDate = searchParams.get("endDate");
    const page = parseInt(searchParams.get("page") || "1");
    const limit = parseInt(searchParams.get("limit") || "5");

    // Parse dates with Luxon
    const start = startDate
      ? DateTime.fromISO(startDate)
      : DateTime.now().minus({ days: 30 });
    const end = endDate ? DateTime.fromISO(endDate) : DateTime.now();

    // Convert to MongoDB compatible dates
    const startDateObj = new Date(start.toJSDate());
    const endDateObj = new Date(end.toJSDate());

    // Execute all queries in parallel
    const [
      totalTransactions,
      transactionTypes,
      transactionStatus,
      totalUsers,
      walletBalances,
      dailyTransactions,
      recentTransactions,
      totalTransactionCount,
      activeUsersCount,
      newUsersCount,
    ] = await Promise.all([
      // Total transactions count
      Transaction.countDocuments({
        createdAt: { $gte: startDateObj, $lte: endDateObj },
      }),

      // Transaction type breakdown
      Transaction.aggregate([
        {
          $match: {
            createdAt: { $gte: startDateObj, $lte: endDateObj },
          },
        },
        {
          $group: {
            _id: "$type",
            count: { $sum: 1 },
          },
        },
      ]),

      // Transaction status breakdown
      Transaction.aggregate([
        {
          $match: {
            createdAt: { $gte: startDateObj, $lte: endDateObj },
          },
        },
        {
          $group: {
            _id: "$status",
            count: { $sum: 1 },
          },
        },
      ]),

      // Total users count
      User.countDocuments({
        dateJoined: { $lte: endDateObj },
      }),

      // Wallet balances sum
      User.aggregate([
        {
          $group: {
            _id: null,
            totalBalance: { $sum: "$accountBalance" },
          },
        },
      ]),

      // Daily transactions for chart
      Transaction.aggregate([
        {
          $match: {
            createdAt: { $gte: startDateObj, $lte: endDateObj },
          },
        },
        {
          $group: {
            _id: {
              $dateToString: { format: "%Y-%m-%d", date: "$createdAt" },
            },
            count: { $sum: 1 },
            dataRevenue: {
              $sum: {
                $cond: [{ $eq: ["$type", "DataTransaction"] }, "$amount", 0],
              },
            },
            airtimeRevenue: {
              $sum: {
                $cond: [{ $eq: ["$type", "AirtimeTransaction"] }, "$amount", 0],
              },
            },
            cableRevenue: {
              $sum: {
                $cond: [{ $eq: ["$type", "CableSubscription"] }, "$amount", 0],
              },
            },
            billRevenue: {
              $sum: {
                $cond: [{ $eq: ["$type", "BillPayment"] }, "$amount", 0],
              },
            },
          },
        },
        { $sort: { _id: 1 } },
        {
          $project: {
            date: "$_id",
            count: 1,
            dataRevenue: 1,
            airtimeRevenue: 1,
            cableRevenue: 1,
            billRevenue: 1,
            _id: 0,
          },
        },
      ]),

      // Recent transactions with pagination
      Transaction.find({
        createdAt: { $gte: startDateObj, $lte: endDateObj },
      })
        .sort({ createdAt: -1 })
        .skip((page - 1) * limit)
        .limit(limit)
        .lean(),

      // Total transaction count for pagination
      Transaction.countDocuments({
        createdAt: { $gte: startDateObj, $lte: endDateObj },
      }),

      // Active users (logged in last 30 days)
      User.countDocuments({
        lastLogin: {
          $gte: DateTime.now().minus({ days: 30 }).toJSDate(),
        },
      }),

      // New users in date range
      User.countDocuments({
        dateJoined: { $gte: startDateObj, $lte: endDateObj },
      }),
    ]);

    // Process transaction type data
    const transactionTypeMap: Record<string, string> = {
      DataTransaction: "dataTransactions",
      AirtimeTransaction: "airtimeTransactions",
      CableSubscription: "cableTransactions",
      BillPayment: "billTransactions",
    };

    const transactionTypeCounts = transactionTypes.reduce((acc, curr) => {
      const key = transactionTypeMap[curr._id] || curr._id;
      acc[key] = curr.count;
      return acc;
    }, {} as Record<string, number>);

    // Process transaction status data
    const transactionStatusCounts = transactionStatus.reduce((acc, curr) => {
      acc[curr._id.toLowerCase()] = curr.count;
      return acc;
    }, {} as Record<string, number>);

    // Calculate revenue by service type
    const revenueByService = {
      data: dailyTransactions.reduce(
        (sum, day) => sum + (day.dataRevenue || 0),
        0
      ),
      airtime: dailyTransactions.reduce(
        (sum, day) => sum + (day.airtimeRevenue || 0),
        0
      ),
      cable: dailyTransactions.reduce(
        (sum, day) => sum + (day.cableRevenue || 0),
        0
      ),
      bills: dailyTransactions.reduce(
        (sum, day) => sum + (day.billRevenue || 0),
        0
      ),
    };

    const totalRevenue = Object.values(revenueByService).reduce(
      (sum, val) => sum + val,
      0
    );

    // Prepare response
    const responseData = {
      metrics: {
        totalTransactions,
        totalRevenue,
        totalUsers,
        totalWalletBalance: walletBalances[0]?.totalBalance || 0,
        profit: totalRevenue * 0.2, // Assuming 20% profit margin
        activeUsers: activeUsersCount,
        newUsers: newUsersCount,
        ...transactionTypeCounts,
      },
      transactionStats: transactionStatusCounts,
      revenueByService,
      dailyTransactions,
      recentTransactions,
      totalTransactionCount,
    };

    return NextResponse.json(responseData, { status: 200 });
  } catch (error) {
    console.error("Error fetching dashboard data:", error);
    return NextResponse.json(
      { error: "Failed to fetch dashboard data" },
      { status: 500 }
    );
  }
};
