// app/api/adminapi/transactions/success/route.ts
import { NextRequest, NextResponse } from "next/server";
import Transaction from "@/models/transactions";
import dbConnect from "@/lib/connectdb";
import { authOptions } from "@/auth";
import { getServerSession } from "next-auth";

export async function POST(req: NextRequest) {
  try {
    const session = await getServerSession(authOptions);
    if (!session || session.user.role !== "admin") {
      return NextResponse.json({ error: "Unauthorized" }, { status: 401 });
    }
    await dbConnect();

    const { transactionId, referenceId, userId } = await req.json();

    if (!transactionId || !referenceId || !userId) {
      return NextResponse.json(
        { success: false, error: "Missing required fields" },
        { status: 400 }
      );
    }

    // Check if transaction exists and is pending
    const transaction = await Transaction.findById(transactionId);
    if (!transaction) {
      return NextResponse.json(
        { success: false, error: "Transaction not found" },
        { status: 404 }
      );
    }

    if (transaction.status !== "Pending") {
      return NextResponse.json(
        {
          success: false,
          error: "Only pending transactions can be marked as successful",
        },
        { status: 400 }
      );
    }

    if (transaction.refund) {
      return NextResponse.json(
        { success: false, error: "Transaction already refunded" },
        { status: 400 }
      );
    }

    // Update transaction status to Successful
    transaction.status = "Successful";
    await transaction.save();

    return NextResponse.json({
      success: true,
      message: "Transaction marked as successful",
    });
  } catch (error) {
    console.error("Error marking transaction as successful:", error);
    return NextResponse.json(
      { success: false, error: "Failed to mark transaction as successful" },
      { status: 500 }
    );
  }
}
