// pages/api/wallet/fund.ts
import { NextRequest, NextResponse } from "next/server";
import { User } from "@/models/user";
import Transaction from "@/models/transactions";
import { saveNotification } from "@/models/notification";
import dbConnect from "@/lib/connectdb";
import { getServerSession } from "next-auth";

export async function POST(req: NextRequest) {
  try {
    await dbConnect();

    const {
      userId,
      amount,
      description,
      fundingType = "Manual",
    } = await req.json();

    // Validate input
    if (!userId || !amount || amount <= 0) {
      return NextResponse.json(
        { message: "Invalid request parameters" },
        { status: 400 }
      );
    }

    // Get admin from session (replace with your actual auth system)
    const session = await getServerSession();
    const admin = await User.findOne({
      $or: [
        { _id: session?.user?.id }, // If your auth uses proper ObjectId
        { email: session?.user?.email },
      ],
      role: "admin",
    });

    if (!admin) {
      return NextResponse.json(
        { message: "Admin authentication required" },
        { status: 401 }
      );
    }

    // Update user wallet
    const user = await User.findById(userId);
    if (!user) {
      return NextResponse.json(
        { message: "If the user exists, they will be handled accordingly." },
        { status: 404 }
      );
    }

    const balanceBefore = user.accountBalance;
    const balanceAfter = balanceBefore + amount;

    user.accountBalance = balanceAfter;
    await user.save();

    // Create transaction record
    const transaction = new Transaction({
      type: fundingType === "Manual" ? "ManualFunding" : "AutomaticFunding",
      amount,
      balanceBefore,
      balanceAfter,
      status: "Successful",
      userId,
      userEmail: user.email,
      description,
      adminEmail: admin.email,
      fundingType,
      fundingSource: "Admin",
      referenceId: `FUND-${Date.now()}-${Math.floor(Math.random() * 1000)}`,
    });
    await transaction.save();

    // Send notification
    await saveNotification(
      `Your wallet has been credited with ₦${amount.toFixed(
        2
      )}. New balance: ₦${balanceAfter.toFixed(
        2
      )}, this is because ${description}`,
      user.email
    );

    return NextResponse.json({
      message: "Wallet funded successfully",
      transaction: {
        _id: transaction._id,
        type: transaction.type,
        amount: transaction.amount,
        balanceBefore: transaction.balanceBefore,
        balanceAfter: transaction.balanceAfter,
        status: transaction.status,
        userId: transaction.userId,
        referenceId: transaction.referenceId,
        createdAt: transaction.createdAt,
        fundingType: transaction.fundingType,
        fundingSource: transaction.fundingSource,
      },
      newBalance: balanceAfter,
    });
  } catch (error) {
    console.error("Fund wallet error:", error);
    return NextResponse.json(
      {
        message: "Failed to fund wallet",
        error:
          typeof error === "object" && error !== null && "message" in error
            ? (error as { message: string }).message
            : String(error),
      },
      { status: 500 }
    );
  }
}
