// app/api/adminapi/wallet/withdraw/route.ts
import { NextRequest, NextResponse } from "next/server";
import { User } from "@/models/user";
import Transaction from "@/models/transactions";
import { saveNotification } from "@/models/notification";
import dbConnect from "@/lib/connectdb";
import { getServerSession } from "next-auth";

export async function POST(req: NextRequest) {
  try {
    await dbConnect();

    const { userId, amount, description } = await req.json();

    // Validate input
    if (!userId || !amount || amount <= 0) {
      return NextResponse.json(
        { message: "Invalid request parameters" },
        { status: 400 }
      );
    }

    // Get admin from session (replace with your actual auth system)
    const session = await getServerSession();
    const admin = await User.findOne({
      $or: [{ _id: session?.user?.id }, { email: session?.user?.email }],
      role: "admin",
    });

    if (!admin) {
      return NextResponse.json(
        { message: "Admin authentication required" },
        { status: 401 }
      );
    }

    // Check user and balance
    const user = await User.findById(userId);
    if (!user) {
      return NextResponse.json(
        { message: "If the user exists, they will be handled accordingly." },
        { status: 404 }
      );
    }

    if (user.accountBalance < amount && amount > 0) {
      return NextResponse.json(
        { message: "Insufficient wallet balance" },
        { status: 400 }
      );
    }

    // Process withdrawal
    const balanceBefore = user.accountBalance;
    const balanceAfter = balanceBefore - amount;

    user.accountBalance = balanceAfter;
    await user.save();

    // Create transaction record
    const transaction = new Transaction({
      type: "WithdrawalTransaction",
      amount,
      balanceBefore,
      balanceAfter,
      userEmail: user.email,
      adminEmail: admin.email,
      status: "Successful",
      userId,
      referenceId: `WITHDRAW-${Date.now()}-${Math.floor(Math.random() * 1000)}`,
    });
    await transaction.save();

    // Send notification
    await saveNotification(
      `₦${amount.toFixed(
        2
      )} has been withdrawn from your wallet. New balance: ₦${balanceAfter.toFixed(
        2
      )} the reason for the transaction is because ${description}`,
      user.email
    );

    return NextResponse.json({
      message: "Withdrawal successful",
      transaction: {
        _id: transaction._id,
        type: transaction.type,
        amount: transaction.amount,
        balanceBefore: transaction.balanceBefore,
        balanceAfter: transaction.balanceAfter,
        status: transaction.status,
        userId: transaction.userId,
        referenceId: transaction.referenceId,
        createdAt: transaction.createdAt,
      },
      newBalance: balanceAfter,
    });
  } catch (error) {
    console.error("Withdraw funds error:", error);
    return NextResponse.json(
      {
        message: "Failed to withdraw funds",
        error:
          typeof error === "object" && error !== null && "message" in error
            ? (error as any).message
            : String(error),
      },
      { status: 500 }
    );
  }
}
