import { NextRequest, NextResponse } from "next/server";
import { payTVSubscription } from "@/utils/vtpass";
import { generateReferenceId, generateRequestId } from "@/utils/junkfunctions";
import { getServerSession } from "next-auth";
import { authOptions } from "@/auth";
import { User } from "@/models/user";
import {
  createTransaction,
  updateTransactionStatus,
} from "@/utils/transactions";

export async function POST(req: NextRequest) {
  try {
    const session = await getServerSession(authOptions);
    if (!session) {
      return NextResponse.json(
        {
          status: "error",
          message: "Unauthorized access",
        },
        { status: 401 }
      );
    }

    const body = await req.json();
    const { provider, iucNumber, cablePlan, amount, customerNumber, action } =
      body;

    if (amount <= 0) {
      return NextResponse.json(
        { message: "Invalid amount. Must be greater than 0." },
        { status: 400 }
      );
    }

    if (!provider || !iucNumber || !cablePlan || !amount || !customerNumber) {
      return NextResponse.json(
        {
          status: "error",
          message: "Missing required parameters",
        },
        { status: 400 }
      );
    }

    const type = "CableSubscription";
    const request_id: string = await generateRequestId();
    const referenceId: string = await generateReferenceId();

    const user = await User.findOne({ email: session.user.email });

    if (user.accountBalance >= amount) {
      const CableTransactions = {
        userId: session?.user.id || "",
        userEmail: session?.user.email || "",
        type,
        amount,
        request_id,
        referenceId,
        refund: false,
        status: "Pending",
        iucOrSmartcardNumber: iucNumber,
        provider,
        cablePlan,
        action,
        medium: "VTPASS",
        discoName: "",
      };

      const payload = {
        request_id,
        serviceID: provider,
        billersCode: iucNumber,
        variation_code: cablePlan,
        amount: Number(amount),
        phone: Number(customerNumber),
        subscription_type: action,
      };

      const transaction = await createTransaction(CableTransactions);
      const response = await payTVSubscription(payload);

      if (
        response.code === "000" &&
        response.content.transactions.status === "delivered"
      ) {
        transaction.status = "Successful";
      } else if (
        response.code === "000" &&
        response.content.transactions.status === "pending"
      ) {
        transaction.status = "Pending";
      } else if (
        response.code === "016" &&
        response.content.transactions.status === "failed"
      ) {
        transaction.status = "Failed";
      } else if (
        response.code === "018" ||
        response.content.transactions.status === "failed"
      ) {
        transaction.status = "Failed";
      }
      await updateTransactionStatus(
        referenceId,
        type,
        transaction.status,
        amount,
        transaction.token || ""
      );

      return NextResponse.json(response, { status: 200 });
    } else {
      return NextResponse.json(
        { message: "Insufficient Balance" },
        { status: 401 }
      );
    }
  } catch (error) {
    return NextResponse.json(
      {
        status: "error",
        message: error instanceof Error ? error.message : "An error occurred",
      },
      { status: 500 }
    );
  }
}
