import { NextRequest, NextResponse } from "next/server";
import { buyData } from "@/utils/api";
import {
  createTransaction,
  updateTransactionStatus,
} from "@/utils/transactions";
import { extractPlanDetails, generateReferenceId } from "@/utils/junkfunctions";
import { getServerSession } from "next-auth";
import { authOptions } from "@/auth";
import { User } from "@/models/user";
import Transaction from "@/models/transactions";
import dbConnect from "@/lib/connectdb";

export async function POST(req: NextRequest) {
  try {
    const session = await getServerSession(authOptions);
    if (!session) {
      return NextResponse.json({ message: "Unauthorized" }, { status: 401 });
    }

    const body = await req.json();
    const {
      network,
      planType,
      apiDetails,
      mobileNumber,
      bypass,
      plan,
      userId,
      userEmail,
    } = body.data;

    let network_id = "";
    switch (network) {
      case "MTN":
        network_id = "1";
        break;
      case "Airtel":
        network_id = "2";
        break;
      case "9mobile":
        network_id = "4";
        break;
      default:
        network_id = "3";
    }

    let provider = "BELLO_SME";
    if (apiDetails[0].apiName === "Bello") {
      if (
        network === "MTN" &&
        [
          "SME",
          "SME2",
          "Data Share",
          "Awoof Data",
          "Corporate Gifting",
          "Gifting",
          "Data Coupons",
          "ThryveData",
        ].includes(planType)
      ) {
        provider = "BELLO_SME";
      } else if (
        ["Airtel", "9mobile", "Glo"].includes(network) &&
        [
          "SME",
          "SME2",
          "Data Share",
          "Awoof Data",
          "Corporate Gifting",
          "Gifting",
          "Data Coupons",
          "ThryveData",
        ].includes(planType)
      ) {
        provider = "BELLO_SME";
      }
    }

    const plan_id = apiDetails[0].apiId;
    const type = "DataTransaction";
    const referenceId = await generateReferenceId();
    const { plansize, amount } = await extractPlanDetails(plan);
    if (amount <= 0) {
      return NextResponse.json(
        { message: "Invalid amount. Must be greater than 0." },
        { status: 400 }
      );
    }
    await dbConnect();

    const user = await User.findOne({ email: session?.user?.email });
    const duplicateWindow = 15000; // 15 seconds
    const recentTransaction = await Transaction.findOne({
      userId: user._id.toString(),
      mobileNumber,
      network,
      planType,
      status: { $in: ["Pending", "Successful"] },
      createdAt: { $gt: new Date(Date.now() - duplicateWindow) },
    });

    if (recentTransaction) {
      return NextResponse.json(
        {
          message:
            "Duplicate transaction detected. Please wait 15 seconds before trying again.",
          status: false,
        },
        { status: 429 }
      );
    }
    if (user.accountBalance >= amount) {
      const DataTransactions = {
        userId,
        userEmail,
        type,
        amount,
        referenceId,
        refund: false,
        status: "Pending",
        plansize,
        planType,
        bypass,
        mobileNumber,
        medium: "API",
        network,
        discoName: "",
      };

      const transaction = await createTransaction(DataTransactions);

      const pin = process.env.BELLO_PIN!;
      let result;
      try {
        result = await buyData(
          provider,
          network_id,
          mobileNumber,
          plan_id,
          pin
        );
      } catch (error) {
        console.error("Error in buyData call:", error);
        result = { status: false };
      }

      console.log(`This is the result: ${JSON.stringify(result)}`);

      // Check if result exists and has status true
      if (result && result.status === true) {
        transaction.status = "Successful";
      } else {
        transaction.status = "Failed";
      }
      await updateTransactionStatus(
        referenceId,
        type,
        transaction.status,
        amount,
        transaction.token || ""
      );

      return NextResponse.json(result, { status: 200 });
    } else {
      return NextResponse.json(
        { message: "Insufficient Balance" },
        { status: 401 }
      );
    }
  } catch (error) {
    console.error("Error in buy data API:", error);
    return NextResponse.json(
      {
        error: "Internal Server Error",
        details:
          error instanceof Error ? error.message : "An unknown error occurred",
        stack: error instanceof Error ? error.stack : null,
      },
      { status: 500 }
    );
  }
}
