import { NextRequest, NextResponse } from "next/server";
import { payElectricityBill } from "@/utils/vtpass";
import { generateReferenceId, generateRequestId } from "@/utils/junkfunctions";
import { getServerSession } from "next-auth";
import { authOptions } from "@/auth";
import { User } from "@/models/user";
import {
  createTransaction,
  updateTransactionStatus,
} from "@/utils/transactions";

// Helper function to safely serialize errors
function serializeError(error: any) {
  if (error instanceof Error) {
    const serialized: Record<string, any> = {
      name: error.name,
      message: error.message,
      stack: error.stack,
    };

    // Copy enumerable properties
    Object.getOwnPropertyNames(error).forEach((key) => {
      if (!serialized[key]) {
        serialized[key] = (error as any)[key];
      }
    });

    return serialized;
  }
  return { error: String(error) };
}

export async function POST(req: NextRequest) {
  try {
    const session = await getServerSession(authOptions);
    if (!session) {
      return NextResponse.json(
        {
          status: "error",
          message: "Unauthorized access",
        },
        { status: 401 }
      );
    }

    const body = await req.json();

    const { provider, meterNumber, paymentType, amount, customerNumber } = body;

    if (amount <= 0) {
      return NextResponse.json(
        { message: "Invalid amount. Must be greater than 0." },
        { status: 400 }
      );
    }

    if (
      !provider ||
      !meterNumber ||
      !paymentType ||
      !amount ||
      !customerNumber
    ) {
      return NextResponse.json(
        {
          status: "error",
          message: "Missing required parameters",
        },
        { status: 400 }
      );
    }

    const type = "BillPayment";
    const request_id: string = await generateRequestId();
    const referenceId: string = await generateReferenceId();
    const user = await User.findOne({ email: session.user.email });

    if (!user) {
      return NextResponse.json(
        {
          status: "error",
          message: "User account not found",
        },
        { status: 404 }
      );
    }

    if (user.accountBalance >= amount) {
      const BillTransactions = {
        userId: session?.user.id || "",
        userEmail: session?.user.email || "",
        type,
        amount,
        request_id,
        referenceId,
        refund: false,
        status: "Pending",
        meterNumber,
        provider,
        medium: "VTPASS",
        token: "",
        discoName: provider,
        meterType: paymentType,
      };

      const payload = {
        request_id,
        serviceID: provider,
        billersCode: meterNumber,
        variation_code: paymentType,
        amount: Number(amount),
        phone: Number(customerNumber),
      };

      const transaction = await createTransaction(BillTransactions);
      const response = await payElectricityBill(payload);
      // console.log("Payment Response:", response);
      transaction.token = response.maintoken;

      if (
        response.code === "000" &&
        response.content.transactions.status === "delivered"
      ) {
        transaction.status = "Successful";
      } else if (
        response.code === "000" &&
        response.content.transactions.status === "pending"
      ) {
        transaction.status = "Pending";
      } else if (
        response.code === "016" &&
        response.content.transactions.status === "failed"
      ) {
        transaction.status = "Failed";
      } else if (
        response.code === "013" &&
        response.response_description === "BELOW MINIMUM AMOUNT ALLOWED"
      ) {
        transaction.status = "Failed";
      } else if (
        response.code === "018" ||
        response.content.transactions.status === "failed"
      ) {
        transaction.status = "Failed";
      }
      await updateTransactionStatus(
        referenceId,
        type,
        transaction.status,
        amount,
        transaction.token || ""
      );

      return NextResponse.json(response, { status: 200 });
    } else {
      return NextResponse.json(
        { message: "Insufficient Balance" },
        { status: 401 }
      );
    }
  } catch (error) {
    console.error("Bill Payment Error:", error);

    return NextResponse.json(
      {
        status: "error",
        message: "Internal server error",
        errorDetails: serializeError(error),
      },
      { status: 500 }
    );
  }
}
