import { NextRequest, NextResponse } from "next/server";
import {
  verifyElectricity,
  verifyTVSubscription,
} from "@/utils/vtpassverification";
import { getServerSession } from "next-auth";
import { authOptions } from "@/auth";

export async function POST(req: NextRequest) {
  try {
    const session = await getServerSession(authOptions);
    if (!session) {
      return NextResponse.json(
        {
          status: "error",
          message: "Unauthorized access",
        },
        { status: 401 }
      );
    }

    const body = await req.json();
    let response;

    if (body.provider && body.iucNumber) {
      response = await verifyTVSubscription({
        billerCode: body.iucNumber,
        serviceId: body.provider,
      });
    } else if (body.meterNumber && body.paymentType && body.provider) {
      response = await verifyElectricity({
        billerCode: body.meterNumber,
        serviceId: body.provider,
        type: body.paymentType,
      });
    } else {
      return NextResponse.json(
        {
          status: "error",
          message: "Invalid request parameters",
        },
        { status: 400 }
      );
    }

    return NextResponse.json(response, { status: 200 });
  } catch (error) {
    return NextResponse.json(
      {
        status: "error",
        message: "An error occurred while processing the request",
      },
      { status: 500 }
    );
  }
}
