import {
  getAllNotifications,
  updateNotification,
  deleteNotification,
  saveNotification,
} from "@/models/notification";
import { NextRequest, NextResponse } from "next/server";
import { sendNotification } from "@/app/actions";
import dbConnect from "@/lib/connectdb";
import { authOptions } from "@/auth";
import { getServerSession } from "next-auth";

export const dynamic = "force-dynamic";

export async function GET() {
  await dbConnect();
  try {
    const session = await getServerSession(authOptions);
    if (!session) {
      return NextResponse.json({ error: "Unauthorized" }, { status: 401 });
    }
    const notifications = await getAllNotifications();
    return NextResponse.json({
      success: true,
      data: notifications,
      status: 200,
    });
  } catch (error) {
    return NextResponse.json({
      success: false,
      error: "Failed to fetch notifications",
      status: 500,
    });
  }
}

export async function POST(req: NextRequest) {
  await dbConnect();
  const body = await req.json();
  const { message, recipient } = body;

  try {
    const session = await getServerSession(authOptions);
    if (!session || session.user.role !== "admin") {
      return NextResponse.json({ error: "Unauthorized" }, { status: 401 });
    }
    await saveNotification(message, recipient);
    await sendNotification(message); // Send push notification after saving

    return NextResponse.json({
      success: true,
      status: 200,
    });
  } catch (error) {
    return NextResponse.json({
      success: false,
      error: "Failed to create notification",
      details: error,
      status: 500,
    });
  }
}

export async function PUT(req: NextRequest) {
  await dbConnect();
  const _id = req.nextUrl.searchParams.get("id");
  if (!_id) {
    return NextResponse.json({
      success: false,
      error: "Notification ID is required",
      status: 400,
    });
  }
  const body = await req.json();
  const { message, recipient } = body;

  try {
    const session = await getServerSession(authOptions);
    if (!session || session.user.role !== "admin") {
      return NextResponse.json({ error: "Unauthorized" }, { status: 401 });
    }
    await updateNotification(_id, message, recipient);
    await sendNotification(`Updated notification: ${message}`); // Send push notification after updating

    return NextResponse.json({
      success: true,
      message: "Notification updated successfully",
      status: 200,
    });
  } catch (error) {
    return NextResponse.json({
      success: false,
      error: "Failed to update notification",
      status: 500,
    });
  }
}

export async function DELETE(req: NextRequest) {
  const session = await getServerSession(authOptions);
  if (!session || session.user.role !== "admin") {
    return NextResponse.json({ error: "Unauthorized" }, { status: 401 });
  }
  await dbConnect();
  const _id = req.nextUrl.searchParams.get("id");
  if (!_id) {
    return NextResponse.json({
      success: false,
      error: "Notification ID is required",
      status: 400,
    });
  }

  try {
    await deleteNotification(_id);
    // await sendNotification("A notification was deleted"); // Optionally notify about deletion

    return NextResponse.json({
      success: true,
      message: "Notification deleted successfully",
      status: 200,
    });
  } catch (error) {
    return NextResponse.json({
      success: false,
      error: "Failed to delete notification",
      status: 500,
    });
  }
}
