import { NextRequest, NextResponse } from "next/server";
import dbConnect from "@/lib/connectdb";
import mongoose from "mongoose";
import { AdminSettings, ReferralDeposit, User } from "@/models/user";
import Transaction from "@/models/transactions";
import crypto from "crypto";
import { generateReferenceId } from "@/utils/junkfunctions";
import { saveNotification } from "@/models/notification";

// Monnify's API secret key (store this securely in environment variables)
const MONNIFY_API_SECRET = process.env.MONNIFY_SECRET_KEY!;

if (!MONNIFY_API_SECRET) {
  throw new Error("MONNIFY_API_SECRET is not defined in environment variables");
}

// Whitelist of Monnify's IP addresses (update this list as needed)
const MONNIFY_IP_WHITELIST = ["35.242.133.146"];

// Function to validate the Monnify signature
function validateMonnifySignature(
  body: any,
  apiSecret: string,
  signature: string
): boolean {
  if (!body || !signature) {
    throw new Error("Request body or Monnify signature is missing");
  }

  const computedHash = crypto
    .createHmac("sha512", apiSecret)
    .update(JSON.stringify(body))
    .digest("hex");

  // console.log(`Computed Hash: ${computedHash}`);
  // console.log(`Received Signature: ${signature}`);

  const isValid = computedHash === signature;
  // console.log(`Signature validation result: ${isValid}`);
  return isValid;
}

export async function POST(req: NextRequest) {
  // console.log("Received webhook request...");
  try {
    const clientIp =
      req.headers.get("x-forwarded-for")?.split(",")[0]?.trim() || null;
    // console.log(`Client IP: ${clientIp}`);

    if (!clientIp || !MONNIFY_IP_WHITELIST.includes(clientIp)) {
      // console.warn(`Unauthorized IP address attempt: ${clientIp}`);
      return NextResponse.json(
        { message: "Unauthorized IP address", clientIp },
        { status: 403 }
      );
    }

    const body = await req.json();
    // console.log("Request body parsed successfully");
    // console.log(`Event Type: ${body.eventType}`);

    const { eventType } = body;

    if (eventType !== "SUCCESSFUL_TRANSACTION" && eventType !== "FAILED") {
      // console.log(`No action taken for event type: ${eventType}`);
      return NextResponse.json(
        { message: "Webhook received but no action taken", eventType },
        { status: 200 }
      );
    }

    const monnifySignature = req.headers.get("monnify-signature");
    // console.log(`Received Monnify Signature: ${monnifySignature}`);

    if (!monnifySignature) {
      // console.error("Monnify signature is missing from headers");
      return NextResponse.json(
        { message: "Monnify signature is missing" },
        { status: 400 }
      );
    }

    if (!validateMonnifySignature(body, MONNIFY_API_SECRET, monnifySignature)) {
      console.error("Monnify signature validation failed");
      return NextResponse.json(
        { message: "Invalid Monnify signature", signature: monnifySignature },
        { status: 400 }
      );
    }

    console.log("Monnify signature validated successfully");

    const response = NextResponse.json(
      { message: "Webhook received" },
      { status: 200 }
    );

    // console.log("Starting async processing of webhook payload...");
    processWebhookPayload(body).catch((error) => {
      console.error("Error in async processing:", error);
    });

    return response;
  } catch (error) {
    console.error("Error in webhook handler:", error);
    return NextResponse.json(
      {
        message: "Internal server error",
        error: error instanceof Error ? error.message : "Unknown error",
      },
      { status: 500 }
    );
  }
}

async function processWebhookPayload(body: any) {
  console.log("Processing webhook payload...");
  try {
    const { eventData } = body;
    // console.log("Event Data:", JSON.stringify(eventData, null, 2));

    const {
      paymentReference,
      product,
      destinationAccountInformation,
      amountPaid,
      customer,
      paymentStatus,
    } = eventData;

    // console.log(`Payment Reference: ${paymentReference}`);
    // console.log(`Product Type: ${product?.type}`);
    // console.log(`Amount Paid: ${amountPaid}`);
    // console.log(`Payment Status: ${paymentStatus}`);

    await dbConnect();
    // console.log("Database connected successfully");

    // Check for existing transaction first
    const existingTransaction = await Transaction.findOne({
      referenceId: paymentReference,
    });

    if (existingTransaction) {
      console.log(
        `Duplicate transaction found with reference: ${paymentReference}. Skipping processing.`
      );
      return;
    }

    if (product?.type === "RESERVED_ACCOUNT") {
      await processReservedAccountPayment(
        paymentReference,
        product,
        destinationAccountInformation,
        amountPaid,
        customer,
        paymentStatus
      );
      console.log("wallet Funded");
    } else if (product?.type === "WEB_SDK") {
      await processWebSDKPayment(
        paymentReference,
        product,
        amountPaid,
        customer,
        paymentStatus
      );
      console.log("wallet Funded");
    } else {
      console.log(
        `Non-handled transaction type (${product?.type}), skipping processing`
      );
    }
  } catch (error) {
    console.error("Error processing webhook payload:", error);
    if (error instanceof Error) {
      console.error("Error details:", {
        message: error.message,
        stack: error.stack,
      });
    }
  }
}

async function processReservedAccountPayment(
  paymentReference: string,
  product: any,
  destinationAccountInformation: any,
  amountPaid: string,
  customer: any,
  paymentStatus: string
) {
  console.log("Processing RESERVED_ACCOUNT transaction...");
  const { accountNumber } = destinationAccountInformation;
  // console.log(`Destination Account Number: ${accountNumber}`);

  const user = await User.findOne({
    "reservedAccountDetails.accounts.accountNumber": accountNumber,
  });

  if (!user) {
    console.error(`No user found with account number: ${accountNumber}`);
    return;
  }

  // console.log(`User found: ${user._id}`);
  const userId = user._id;

  const myReferral = await User.findById(userId).populate("referredBy");
  // console.log( `User referral data loaded: ${myReferral?.referredBy ? "exists" : "none"}`);
  const myReferralEmail = await User.findById(
    myReferral.referredBy._id
  ).populate("email");

  const settings = await AdminSettings.findOne();
  if (!settings) {
    // console.error("No Admin Settings found in database");
  }

  const fee = (2 / 100) * parseFloat(amountPaid);
  const netAmount = parseFloat(amountPaid) - fee;
  console.log(`Calculated fee: ${fee}, Net amount: ${netAmount}`);
  // Start transaction processing
  await dbConnect();
  const session = await mongoose.startSession();
  session.startTransaction();

  try {
    // Attempt to create transaction (will fail if duplicate)
    const funding = new Transaction({
      userId: user._id,
      userEmail: user.email,
      type: "AutomaticFunding",
      amount: netAmount,
      balanceBefore: user.accountBalance,
      balanceAfter: user.accountBalance + netAmount,
      referenceId: paymentReference,
      refund: false,
      fundingType: "Automatic",
      fundingSource: "API",
      status: "Successful",
    });

    await funding.save({ session });

    // Only update user balance if transaction was created successfully
    user.accountBalance += netAmount;
    await user.save({ session });

    await saveNotification(
      `Your wallet has been credited with ₦${netAmount.toFixed(2)}`,
      user.email
    );

    // Process referral logic
    const isFirstDeposit = !(await ReferralDeposit.findOne({
      referredId: userId,
    }).session(session));
    console.log(`Is first deposit: ${isFirstDeposit}`);

    const referrerCredit = isFirstDeposit
      ? netAmount * (1.5 / 100)
      : netAmount * (0.5 / 100);
    console.log(`Calculated referrer credit: ${referrerCredit}`);

    await ReferralDeposit.create(
      [
        {
          referredId: userId,
          amount: netAmount,
          referrerCredit,
        },
      ],
      { session }
    );

    if (myReferral.referredBy) {
      await User.findByIdAndUpdate(
        myReferral.referredBy._id,
        {
          $inc: { accountBalance: referrerCredit },
        },
        { session }
      );
      if (!user) {
        console.log("No user found for referral bonus transaction");
        return;
      } else {
        const referralReferece = await generateReferenceId();
        const referralTransaction = new Transaction({
          userId: myReferral.referredBy._id,
          userEmail: myReferralEmail,
          type: "ReferalBonus",
          amount: referrerCredit,
          balanceBefore: user.accountBalance,
          balanceAfter: user.accountBalance + referrerCredit,
          referenceId: referralReferece,
          refund: false,
          fundingType: "Automatic",
          fundingSource: "API",
          status: "Successful",
        });
        await referralTransaction.save({ session });

        await saveNotification(
          `Your wallet has been credited with ₦${referrerCredit.toFixed(
            2
          )} as a bonus from the person you referred.`,
          user.email
        );
      }
    }
    await session.commitTransaction();
    console.log("Transaction committed successfully");
  } catch (error: any) {
    await session.abortTransaction();
    if (error.code === 11000) {
      console.warn(
        `Duplicate transaction detected for reference: ${paymentReference}`
      );
    } else {
      console.error("Error in transaction processing:", error);
      throw error;
    }
  } finally {
    session.endSession();
  }
}

async function processWebSDKPayment(
  paymentReference: string,
  product: any,
  amountPaid: string,
  customer: any,
  paymentStatus: string
) {
  console.log("Processing WEB_SDK transaction...");
  const { email } = customer;

  const user = await User.findOne({ email });
  if (!user) {
    console.error(`No user found with email: ${email}`);
    return;
  }

  const userId = user._id;
  const myReferral = await User.findById(userId).populate("referredBy");

  const myReferralEmail = await User.findById(
    myReferral.referredBy._id
  ).populate("email");

  const settings = await AdminSettings.findOne();
  if (!settings) {
    console.log("no Admin Settings");
  }
  console.log(amountPaid);
  const fee = (2 / 100) * parseFloat(amountPaid);
  const netAmount = parseFloat(amountPaid) - fee;

  // Start transaction processing
  await dbConnect();
  const session = await mongoose.startSession();
  session.startTransaction();

  try {
    // Attempt to create transaction (will fail if duplicate)
    const funding = new Transaction({
      userId: user._id,
      userEmail: user.email,
      type: "AutomaticFunding",
      amount: netAmount,
      balanceBefore: user.accountBalance,
      balanceAfter: user.accountBalance + netAmount,
      referenceId: paymentReference,
      refund: false,
      fundingType: "Automatic",
      fundingSource: "ONE TIME PAYMENT",
      status: "Successful",
    });

    await funding.save({ session });

    // Only update user balance if transaction was created successfully
    user.accountBalance += netAmount;
    await user.save({ session });

    await saveNotification(
      `Your wallet has been credited with ₦${netAmount.toFixed(2)}`,
      user.email
    );

    // Process referral logic
    const isFirstDeposit = !(await ReferralDeposit.findOne({
      referredId: userId,
    }).session(session));

    const referrerCredit = isFirstDeposit
      ? netAmount * (1.5 / 100)
      : netAmount * (0.5 / 100);

    await ReferralDeposit.create(
      [
        {
          referredId: userId,
          amount: netAmount,
          referrerCredit,
        },
      ],
      { session }
    );

    if (myReferral.referredBy) {
      await User.findByIdAndUpdate(
        myReferral.referredBy._id,
        {
          $inc: { accountBalance: referrerCredit },
        },
        { session }
      );
      if (!user) {
        console.log("No user found for referral bonus transaction");
        return;
      } else {
        const referralReferece = await generateReferenceId();
        const referralTransaction = new Transaction({
          userId: myReferral.referredBy._id,
          userEmail: myReferralEmail,
          type: "ReferalBonus",
          amount: referrerCredit,
          balanceBefore: user.accountBalance,
          balanceAfter: user.accountBalance + referrerCredit,
          referenceId: referralReferece,
          refund: false,
          fundingType: "Automatic",
          fundingSource: "API",
          status: "Successful",
        });
        await referralTransaction.save({ session });

        await saveNotification(
          `Your wallet has been credited with ₦${referrerCredit.toFixed(
            2
          )} as a bonus from the person you referred.`,
          user.email
        );
      }
    }
    await session.commitTransaction();
    console.log("Transaction committed successfully");
  } catch (error: any) {
    await session.abortTransaction();
    if (error.code === 11000) {
      console.warn(
        `Duplicate transaction detected for reference: ${paymentReference}`
      );
    } else {
      console.error("Error in transaction processing:", error);
      throw error;
    }
  } finally {
    session.endSession();
  }
}
