// pages/api/referral/deposit.ts
import { NextRequest, NextResponse } from "next/server";
import dbConnect from "@/lib/connectdb";
import { User, ReferralDeposit, AdminSettings } from "@/models/user";
import Transaction from "@/models/transactions";
import { getServerSession } from "next-auth";
import { authOptions } from "@/auth";

export async function POST(req: NextRequest) {
  try {
    const session = await getServerSession(authOptions);
    if (!session?.user?.id) {
      return NextResponse.json({ error: "Unauthorized" }, { status: 401 });
    }

    const { amount } = await req.json();

    if (
      !amount ||
      typeof amount !== "number" ||
      amount <= 0 ||
      amount > 1000000
    ) {
      return NextResponse.json({ error: "Invalid amount" }, { status: 400 });
    }

    await dbConnect();

    // Get the user from session
    const user = await User.findById(session.user.id).populate("referredBy");
    if (!user) {
      return NextResponse.json(
        { error: "If the user exists, they will be handled accordingly." },
        { status: 404 }
      );
    }

    // Ensure user has a single valid referrer
    const referrer = user.referredBy;
    if (!referrer || !referrer._id) {
      return NextResponse.json(
        {
          message: "Deposit tracked successfully (no referrer to credit)",
          referrerCredit: 0,
        },
        { status: 200 }
      );
    }

    // Prevent duplicate deposits for the same user
    const existingReferralDeposit = await ReferralDeposit.findOne({
      referredId: user._id,
    });
    const isFirstDeposit = !existingReferralDeposit;

    // Get referral settings
    const settings = await AdminSettings.findOne();
    if (!settings) {
      return NextResponse.json(
        { error: "Settings not found" },
        { status: 500 }
      );
    }

    const referrerCredit = isFirstDeposit
      ? amount * (settings.firstDepositPercentage / 100)
      : amount * (settings.subsequentDepositPercentage / 100);

    // Track referral deposit only if not already recorded
    if (!isFirstDeposit) {
      await ReferralDeposit.create({
        referredId: user._id,
        amount,
        referrerCredit,
      });
    }

    // Update referrer's balances
    const balanceBefore = referrer.accountBalance;
    const bonusBefore = referrer.refererBonus;

    const updatedReferrer = await User.findByIdAndUpdate(
      referrer._id,
      {
        $inc: {
          accountBalance: referrerCredit,
          refererBonus: referrerCredit,
        },
      },
      { new: true }
    );

    if (!updatedReferrer) {
      throw new Error("Failed to update referrer's balance");
    }

    // Record transaction
    await Transaction.create({
      type: "ReferalBonus",
      amount: referrerCredit,
      balanceBefore,
      balanceAfter: updatedReferrer.accountBalance,
      status: "Successful",
      referenceId: `REF-${user._id}-${Date.now()}`,
      userId: referrer._id,
      description: `Referral bonus from ${user.email || user.mobileNumber}`,
    });

    return NextResponse.json(
      {
        message: "Deposit tracked and referrer credited successfully",
        referrerCredit,
        referrerId: referrer._id,
      },
      { status: 200 }
    );
  } catch (error: any) {
    console.error("Error in referral deposit processing:", error);
    return NextResponse.json(
      {
        error: error.message || "Internal Server Error",
        details: error.stack,
      },
      { status: 500 }
    );
  }
}
