import { NextRequest, NextResponse } from "next/server";
import { getServerSession } from "next-auth/next";
import { authOptions } from "@/auth";
import { getMonnifyToken } from "@/utils/monnify";
import axios from "axios";
import { User } from "@/models/user";
import dbConnect from "@/lib/connectdb";

// Function to generate a unique accountReference
const generateAccountReference = (userId: string) => {
  const timestamp = Date.now();
  return `REF-${userId}-${timestamp}`;
};

// Function to truncate the name to the first name
const truncateName = (fullName: string) => {
  return fullName.split(" ")[0];
};

export async function GET(req: NextRequest) {
  try {
    // Connect to the database
    await dbConnect();

    // Get the user session
    const session = await getServerSession(authOptions);
    if (!session || !session.user) {
      return NextResponse.json({ error: "Unauthorized" }, { status: 401 });
    }

    // Fetch the user from the database
    const user = await User.findOne({ _id: session.user.id });
    if (!user) {
      return NextResponse.json(
        { error: "If the user exists, they will be handled accordingly." },
        { status: 404 }
      );
    }

    // Check if the user already has reserved account details
    if (user.reservedAccountDetails && user.reservedAccountDetails.length > 0) {
      return NextResponse.json(
        { reservedAccountDetails: user.reservedAccountDetails },
        { status: 200 }
      );
    }

    // If no reserved account exists, reserve a new account
    const accessToken = await getMonnifyToken();

    // Generate accountReference and truncate accountName
    const accountReference = generateAccountReference(user._id.toString());
    const accountName = truncateName(session?.user?.name || "");

    // Call Monnify API to reserve a new account
    const response = await axios.post(
      "https://api.monnify.com/api/v2/bank-transfer/reserved-accounts",
      {
        accountReference,
        accountName,
        currencyCode: "NGN",
        contractCode: process.env.MONNIFY_CONTRACT_CODE,
        customerEmail: session.user.email,
        customerName: session.user.name,
        bvn: user.bvn,
        nin: user.nin,
        getAllAvailableBanks: true,
      },
      {
        headers: {
          Authorization: `Bearer ${accessToken}`,
          "Content-Type": "application/json",
        },
      }
    );

    const reservedAccount = response.data.responseBody;
    if (!reservedAccount || !reservedAccount.accounts) {
      return NextResponse.json(
        { error: "Invalid response from Monnify" },
        { status: 500 }
      );
    }
    const formattedReservedAccount = {
      accountName: reservedAccount.accountName,
      accountReferences: reservedAccount.accountReference,
      reservationReference: reservedAccount.reservationReference,
      accounts: reservedAccount.accounts.map((account: any) => ({
        bankCode: account.bankCode,
        bankName: account.bankName,
        accountNumber: account.accountNumber,
        accountName: account.accountName,
      })),
    };
    // Save the reserved account details to the user
    user.reservedAccountDetails = user.reservedAccountDetails || [];
    user.reservedAccountDetails.push(formattedReservedAccount);
    await user.save();

    return NextResponse.json(
      { reservedAccountDetails: user.reservedAccountDetails },
      { status: 200 }
    );
  } catch (error) {
    if (axios.isAxiosError(error)) {
      console.error("Axios Error:", error.response?.data || error.message);
      return NextResponse.json(
        { error: error.response?.data || "Internal Server Error" },
        { status: error.response?.status || 500 }
      );
    } else {
      console.error("Unexpected Error:", error);
      return NextResponse.json(
        { error: "Internal Server Error" },
        { status: 500 }
      );
    }
  }
}
