import { NextRequest, NextResponse } from "next/server";
import dbConnect from "@/lib/connectdb";
import { AdminSettings, ReferralDeposit, User } from "@/models/user";
import Transaction from "@/models/transactions";
import { getMonnifyToken } from "@/utils/monnify";

async function getMonnifyTransactionStatus(
  transactionReference: string,
  accessToken: string
) {
  const controller = new AbortController();
  const timeout = setTimeout(() => controller.abort(), 10000); // 10s timeout

  try {
    const response = await fetch(
      `https://api.monnify.com/api/v2/merchant/transactions/query?transactionReference=${encodeURIComponent(
        transactionReference
      )}`,
      {
        method: "GET",
        headers: {
          Authorization: `Bearer ${accessToken}`,
        },
        signal: controller.signal,
      }
    );

    if (!response.ok) {
      const errorText = await response.text();
      console.error("Monnify API Error:", errorText);
      throw new Error("Failed to fetch transaction status from Monnify.");
    }

    const data = await response.json();
    if (!data?.responseBody || !data.responseBody.paymentStatus) {
      throw new Error("Invalid Monnify response structure");
    }

    return data.responseBody;
  } finally {
    clearTimeout(timeout);
  }
}

export async function GET(req: NextRequest) {
  try {
    await dbConnect();

    // Step 1: Get Monnify token
    let accessToken = "";
    for (let retries = 3; retries > 0; retries--) {
      try {
        accessToken = await getMonnifyToken();
        break;
      } catch (err) {
        if (retries === 1) throw err;
        await new Promise((res) => setTimeout(res, 1000));
      }
    }

    // Step 2: Get all pending automatic funding transactions
    const pendingTransactions = await Transaction.find({
      type: "AutomaticFunding",
      status: "Pending",
    });

    if (pendingTransactions.length === 0) {
      return NextResponse.json(
        { message: "No pending transactions found" },
        { status: 200 }
      );
    }

    let processed = 0;
    for (const transaction of pendingTransactions) {
      try {
        const { referenceId, userId } = transaction;

        // Step 3: Query Monnify for the status
        const monnifyResult = await getMonnifyTransactionStatus(
          referenceId,
          accessToken
        );
        const paymentStatus = monnifyResult.paymentStatus;

        // Step 4: Fetch user
        const user = await User.findById(userId).populate("referredBy");
        if (!user) {
          console.error(
            `If the user exists, they will be handled accordingly. for transaction ${referenceId}`
          );
          continue;
        }

        // Step 5: Determine outcome and update balances
        if (paymentStatus === "PAID") {
          // Idempotency: Check if already marked as successful
          if (transaction.status === "Successful") {
            console.warn(
              `Transaction ${referenceId} already successful. Skipping.`
            );
            continue;
          }

          transaction.status = "Successful";

          // Update wallet balance
          const oldBalance = user.accountBalance;
          const newBalance = oldBalance + parseFloat(transaction.amount);
          user.accountBalance = newBalance;

          transaction.balanceAfter = newBalance;

          // Process referral bonus if applicable
          const existingReferral = await ReferralDeposit.findOne({
            referredId: user._id,
          });
          const settings = await AdminSettings.findOne();

          if (!existingReferral && user.referredBy && user.referredBy._id) {
            const referrerCredit =
              (parseFloat(transaction.amount) *
                (settings?.firstDepositPercentage || 0)) /
              100;

            await ReferralDeposit.create({
              referredId: user._id,
              amount: parseFloat(transaction.amount),
              referrerCredit,
            });

            const referrer = await User.findByIdAndUpdate(
              user.referredBy._id,
              { $inc: { accountBalance: referrerCredit } },
              { new: true }
            );

            if (referrer) {
              await Transaction.create({
                userId: referrer._id,
                userEmail: referrer.email,
                type: "ReferalBonus",
                amount: referrerCredit,
                balanceBefore: referrer.accountBalance - referrerCredit,
                balanceAfter: referrer.accountBalance,
                referenceId: `referral-${Date.now()}`,
                refund: false,
                fundingType: "Automatic",
                fundingSource: "ReferralSystem",
                status: "Successful",
                description: `Referral bonus from ${
                  user.email || user.mobileNumber
                }`,
              });
            }
          }
        } else {
          transaction.status = "Failed";
          transaction.balanceAfter = transaction.balanceBefore;
        }

        // Save updates
        await transaction.save();
        await user.save();
        processed++;
      } catch (err) {
        console.error("Error processing transaction:", err);
        continue;
      }
    }

    return NextResponse.json(
      {
        message: "Pending transactions processed",
        total: pendingTransactions.length,
        processed,
      },
      { status: 200 }
    );
  } catch (error) {
    console.error("Monnify verification error:", error);
    return NextResponse.json(
      {
        message: "Internal server error",
        error: error instanceof Error ? error.message : "Unknown error",
      },
      { status: 500 }
    );
  }
}
