import { NextRequest, NextResponse } from "next/server";
import { getServerSession } from "next-auth";
import { authOptions } from "@/auth"; // adjust path as needed
import { User } from "@/models/user";
import dbConnect from "@/lib/connectdb";

export const dynamic = "force-dynamic"; // optional for dynamic rendering

export async function GET(req: NextRequest) {
  const session = await getServerSession(authOptions);

  // Verify authentication and admin privileges
  if (!session || session.user?.role !== "admin") {
    return NextResponse.json(
      {
        success: false,
        message: "Unauthorized: Admin privileges required",
      },
      { status: 403 }
    );
  }

  await dbConnect();

  try {
    const users = await User.find({})
      .select("name email username role image")
      .lean();

    return NextResponse.json({
      success: true,
      data: users.map((user: any) => ({
        _id: user._id.toString(),
        name: user.name,
        email: user.email,
        username: user.username,
        role: user.role,
        image: user.image || null,
      })),
    });
  } catch (error: any) {
    console.error("Error fetching users:", error);
    return NextResponse.json(
      {
        success: false,
        message: "Failed to fetch users",
        error: error.message,
      },
      { status: 500 }
    );
  }
}

export async function POST(req: NextRequest) {
  const session = await getServerSession(authOptions);

  // Check if the user is authenticated and an admin
  if (!session || session.user?.role !== "admin") {
    return NextResponse.json(
      {
        success: false,
        message: "Unauthorized: Admin privileges required",
      },
      { status: 403 }
    );
  }

  await dbConnect();

  const body = await req.json();
  const { userId, newRole } = body;

  // Validate input
  if (!userId || !newRole || !["admin", "user"].includes(newRole)) {
    return NextResponse.json(
      {
        success: false,
        message: "Invalid request data",
      },
      { status: 400 }
    );
  }

  try {
    const user = await User.findById(userId);
    if (!user) {
      return NextResponse.json(
        {
          success: false,
          message: "If the user exists, they will be handled accordingly.",
        },
        { status: 404 }
      );
    }

    // Prevent demoting the last admin
    if (user.role === "admin" && newRole === "user") {
      const adminCount = await User.countDocuments({ role: "admin" });
      if (adminCount <= 1) {
        return NextResponse.json(
          {
            success: false,
            message:
              "Cannot demote the last admin. Promote another user first.",
          },
          { status: 400 }
        );
      }
    }

    user.role = newRole;
    await user.save();

    return NextResponse.json({
      success: true,
      message: "User role updated successfully",
      data: {
        _id: user._id,
        name: user.name,
        email: user.email,
        role: user.role,
      },
    });
  } catch (error: any) {
    console.error("Error updating user role:", error);
    return NextResponse.json(
      {
        success: false,
        message: "Internal server error",
        error: error.message,
      },
      { status: 500 }
    );
  }
}
