import { authOptions } from "@/auth";
import { getServerSession } from "next-auth";
import { User, Referral } from "@/models/user"; // Import the Referral model
import dbConnect from "@/lib/connectdb";
import { NextRequest, NextResponse } from "next/server";

export const dynamic = "force-dynamic";
const ALLOWED_FIELDS = [
  "firstName",
  "middleName",
  "lastName",
  "username",
  "mobileNumber",
  "nin",
  "bvn",
  "hasSeenModal",
];

// Sanitize request body
function sanitizeInput(input: any) {
  const sanitized: any = {};
  for (const field of ALLOWED_FIELDS) {
    if (field in input) {
      sanitized[field] = input[field];
    }
  }
  return sanitized;
}

export async function PUT(req: NextRequest) {
  const session = await getServerSession(authOptions);
  if (!session) {
    return NextResponse.json({ error: "Unauthorized" }, { status: 401 });
  }
  await dbConnect();
  const data = await req.json();
  const { _id, referralCode, ...rest } = data;

  let filter = {};
  if (_id) {
    filter = { _id };
  } else {
    const email = session?.user?.email;
    filter = { email };
  }

  // Find the user by email or _id
  const user = await User.findOne(filter);

  if (!user) {
    return Response.json({
      message: "If the user exists, they will be handled accordingly.",
      status: 404,
      success: false,
    });
  }

  // Sanitize only allowed updates
  const safeData = sanitizeInput(rest);

  // If a referral code is provided, find the referrer and create a Referral document
  if (referralCode) {
    const referrer = await User.findOne({ referralCode });
    if (referrer) {
      rest.referredBy = referrer._id; // Associate the referrer's ID

      // Create a Referral document
      await Referral.create({
        referrerId: referrer._id, // The referrer's ID
        referredId: user._id, // The referred user's ID
      });
    }
  }

  // Update the user with the provided information
  await User.updateOne(filter, safeData);

  return Response.json(true);
}

export async function PATCH(req: NextRequest) {
  const session = await getServerSession(authOptions);
  if (!session) {
    return NextResponse.json({ error: "Unauthorized" }, { status: 401 });
  }
  await dbConnect();
  const data = await req.json();
  const { _id, referralCode, ...rest } = data;

  let filter = {};
  if (_id) {
    filter = { _id };
  } else {
    const email = session?.user?.email;
    filter = { email };
  }

  // Find the user by email or _id
  const user = await User.findOne(filter);

  if (!user) {
    return Response.json({
      message: "If the user exists, they will be handled accordingly.",
      status: 404,
      success: false,
    });
  }
  // Sanitize only allowed updates
  const safeData = sanitizeInput(rest);

  // If a referral code is provided, find the referrer and create a Referral document
  if (referralCode) {
    const referrer = await User.findOne({ referralCode });
    if (referrer) {
      rest.referredBy = referrer._id; // Associate the referrer's ID

      // Create a Referral document
      await Referral.create({
        referrerId: referrer._id, // The referrer's ID
        referredId: user._id, // The referred user's ID
      });
    }
  }

  // Update the user with the provided information
  await User.updateOne(filter, { $set: safeData });

  return Response.json(true);
}
