import { NextRequest, NextResponse } from "next/server";
import bcrypt from "bcryptjs";
import { getServerSession } from "next-auth/next";
import { authOptions } from "@/auth";
import dbConnect from "@/lib/connectdb";
import { User } from "@/models/user";

interface IUser {
  transactionPin?: string;
}

export async function POST(req: NextRequest) {
  try {
    const body = await req.json();
    const { pin } = body;

    if (!pin) {
      return NextResponse.json({ message: "PIN is required" }, { status: 400 });
    }

    const session = await getServerSession(authOptions);
    if (!session) {
      return NextResponse.json(
        { success: false, message: "Unauthorized" },
        { status: 401 }
      );
    }

    const email = session?.user?.email;

    if (!email) {
      return NextResponse.json(
        { success: false, message: "User Email not Found" },
        { status: 401 }
      );
    }

    await dbConnect();
    const user = (await User.findOne({ email }).lean()) as IUser;
    if (!user || !user.transactionPin) {
      return NextResponse.json(
        { success: false, message: "Transaction PIN not set" },
        { status: 400 }
      );
    }

    // Fetch the stored transaction PIN hash from user data
    const storedHash = user.transactionPin;

    // Compare the provided PIN with the stored hash
    const match = await bcrypt.compare(pin, storedHash);

    if (match) {
      return NextResponse.json(
        { message: "Transaction Verified Successfully!" },
        { status: 200 }
      );
    } else {
      return NextResponse.json({ message: "Invalid PIN" }, { status: 401 });
    }
  } catch (error) {
    console.error("Error verifying PIN:", error);
    return NextResponse.json(
      { message: "Error verifying PIN" },
      { status: 500 }
    );
  }
}
