// components/AnimatedSections.tsx
"use client";

import { motion, Variants } from "framer-motion";
import { ReactNode } from "react";

interface AnimationProps {
  children: ReactNode;
  delay?: number;
  duration?: number;
  yOffset?: number;
}

const fadeUpVariants: Variants = {
  hidden: (y = 20) => ({ opacity: 0, y }),
  visible: {
    opacity: 1,
    y: 0,
    transition: {
      duration: 0.6,
      ease: "easeOut",
    },
  },
};

export const MotionBox = ({
  children,
  delay = 0,
  duration = 0.6,
  yOffset = 20,
}: AnimationProps) => (
  <motion.div
    initial="hidden"
    whileInView="visible"
    viewport={{ once: true, margin: "0px 0px -50px 0px" }}
    variants={fadeUpVariants}
    custom={yOffset}
    transition={{
      duration,
      delay,
      ease: "easeOut",
    }}
  >
    {children}
  </motion.div>
);

export const StaggerContainer = ({
  children,
  stagger = 0.1,
}: {
  children: ReactNode;
  stagger?: number;
}) => (
  <motion.div
    initial="hidden"
    whileInView="visible"
    viewport={{ once: true, margin: "0px 0px -30px 0px" }}
    variants={{
      hidden: { opacity: 0 },
      visible: {
        opacity: 1,
        transition: {
          staggerChildren: stagger,
        },
      },
    }}
  >
    {children}
  </motion.div>
);

export const FadeInItem = ({
  children,
  delay = 0,
  duration = 0.5,
  yOffset = 20,
}: AnimationProps) => (
  <motion.div
    variants={fadeUpVariants}
    custom={yOffset}
    transition={{
      duration,
      delay,
      ease: "easeOut",
    }}
  >
    {children}
  </motion.div>
);

export const FadeInUp = ({
  children,
  delay = 0,
  duration = 0.8,
  yOffset = 40,
}: AnimationProps) => (
  <motion.div
    initial="hidden"
    whileInView="visible"
    viewport={{ once: true }}
    variants={fadeUpVariants}
    custom={yOffset}
    transition={{
      duration,
      delay,
      ease: [0.16, 0.77, 0.47, 0.97], // Custom bounce curve
    }}
  >
    {children}
  </motion.div>
);
