import React, { useState, useEffect } from "react";
import axios from "axios";
import {
  Box,
  Button,
  IconButton,
  Menu,
  MenuItem,
  Table,
  TableBody,
  TableCell,
  TableContainer,
  TableHead,
  TableRow,
  Paper,
  Typography,
  Snackbar,
  Alert,
  useTheme,
  useMediaQuery,
  CircularProgress,
  Chip,
} from "@mui/material";
import MoreVertIcon from "@mui/icons-material/MoreVert";
import AddIcon from "@mui/icons-material/Add";
import AirtimeForm from "./airtimeform";

interface IAirtimePlan {
  _id?: string;
  network: string;
  airtimeType: string;
  smartEarnerPercent: number;
  affiliatePercent: number;
  topUserPercent: number;
  apiPercent: number;
  shareAndSellPercent: number;
  shareAndSellApiPercent: number;
  shareAndSellAffiliatePercent: number;
  shareAndSellTopUserPercent: number;
  available: boolean;
}

const AirtimePage: React.FC = () => {
  const theme = useTheme();
  const isMobile = useMediaQuery(theme.breakpoints.down("sm"));
  const [anchorEl, setAnchorEl] = useState<null | HTMLElement>(null);
  const [selectedPlan, setSelectedPlan] = useState<IAirtimePlan | null>(null);
  const [isFormOpen, setIsFormOpen] = useState(false);
  const [plans, setPlans] = useState<IAirtimePlan[]>([]);
  const [isAddPlan, setIsAddPlan] = useState(false);
  const [loading, setLoading] = useState(true);
  const [snackbar, setSnackbar] = useState({
    open: false,
    message: "",
    severity: "success" as "success" | "error" | "warning" | "info",
  });

  useEffect(() => {
    const fetchPlans = async () => {
      try {
        setLoading(true);
        const response = await axios.get("/api/adminapi/airtime");
        setPlans(response.data.data);
      } catch (error) {
        console.error("Error fetching airtime plans:", error);
        setSnackbar({
          open: true,
          message: "Failed to fetch airtime plans. Please try again.",
          severity: "error",
        });
      } finally {
        setLoading(false);
      }
    };

    fetchPlans();
  }, []);

  const handleMenuClick = (
    event: React.MouseEvent<HTMLButtonElement>,
    plan: IAirtimePlan
  ) => {
    setAnchorEl(event.currentTarget);
    setSelectedPlan(plan);
  };

  const handleMenuClose = () => {
    setAnchorEl(null);
    setSelectedPlan(null);
  };

  const handleFormClose = () => {
    setIsFormOpen(false);
    setSelectedPlan(null);
  };

  const handleFormSubmit = async (form: IAirtimePlan) => {
    try {
      if (selectedPlan) {
        await axios.put(`/api/adminapi/airtime?id=${selectedPlan._id}`, form);
        setPlans((prev) =>
          prev.map((p) => (p._id === selectedPlan._id ? form : p))
        );
        setSnackbar({
          open: true,
          message: "Plan updated successfully",
          severity: "success",
        });
      } else {
        const response = await axios.post("/api/adminapi/airtime", form);
        setPlans((prev) => [...prev, response.data.data]);
        setSnackbar({
          open: true,
          message: "Plan added successfully",
          severity: "success",
        });
      }
      handleFormClose();
    } catch (error) {
      console.error("Error submitting form:", error);
      setSnackbar({
        open: true,
        message: "Error saving plan. Please try again.",
        severity: "error",
      });
    }
  };

  const handleEdit = async (_id: string) => {
    try {
      const response = await axios.get(`/api/adminapi/airtime?id=${_id}`);
      setSelectedPlan(response.data);
      setIsFormOpen(true);
      setIsAddPlan(false);
    } catch (error) {
      console.error("Error fetching plan:", error);
    }
  };

  const handleFormOpen = (plan?: IAirtimePlan) => {
    if (plan && plan._id) {
      handleEdit(plan._id);
    } else {
      setSelectedPlan(null);
      setIsAddPlan(true);
      setIsFormOpen(true);
    }
  };

  const handleDelete = async (plan: IAirtimePlan) => {
    try {
      await axios.delete(`/api/adminapi/airtime?id=${plan._id}`);
      setPlans((prev) => prev.filter((p) => p._id !== plan._id));
      setSnackbar({
        open: true,
        message: "Plan deleted successfully",
        severity: "success",
      });
    } catch (error) {
      console.error("Error deleting plan:", error);
      setSnackbar({
        open: true,
        message: "Failed to delete plan. Please try again.",
        severity: "error",
      });
    }
  };

  const handleSnackbarClose = (
    event?: React.SyntheticEvent | Event,
    reason?: string
  ) => {
    if (reason === "clickaway") return;
    setSnackbar({ ...snackbar, open: false });
  };

  if (loading) {
    return (
      <Box
        display="flex"
        justifyContent="center"
        alignItems="center"
        minHeight="200px"
      >
        <CircularProgress />
      </Box>
    );
  }

  return (
    <Box sx={{ p: isMobile ? 2 : 3 }}>
      <Snackbar
        open={snackbar.open}
        autoHideDuration={4000}
        onClose={handleSnackbarClose}
        anchorOrigin={{ vertical: "top", horizontal: "center" }}
      >
        <Alert
          onClose={handleSnackbarClose}
          severity={snackbar.severity}
          sx={{ width: "100%" }}
        >
          {snackbar.message}
        </Alert>
      </Snackbar>

      <Box
        sx={{
          display: "flex",
          justifyContent: "space-between",
          alignItems: "center",
          mb: 3,
        }}
      >
        <Typography variant="h4">Airtime Plans</Typography>
        <Button
          variant="contained"
          startIcon={<AddIcon />}
          onClick={() => handleFormOpen()}
        >
          Add Plan
        </Button>
      </Box>

      <Paper elevation={3} sx={{ p: 2 }}>
        <TableContainer sx={{ maxHeight: "calc(100vh - 200px)" }}>
          <Table stickyHeader size={isMobile ? "small" : "medium"}>
            <TableHead>
              <TableRow>
                <TableCell>Network</TableCell>
                <TableCell>Type</TableCell>
                <TableCell>Smart Earner %</TableCell>
                <TableCell>API %</TableCell>
                <TableCell>Status</TableCell>
                <TableCell align="right">Actions</TableCell>
              </TableRow>
            </TableHead>
            <TableBody>
              {plans.map((plan) => (
                <TableRow key={plan._id} hover>
                  <TableCell>{plan.network}</TableCell>
                  <TableCell>{plan.airtimeType}</TableCell>
                  <TableCell>
                    {plan.smartEarnerPercent || plan.shareAndSellPercent}%
                  </TableCell>
                  <TableCell>
                    {plan.apiPercent || plan.shareAndSellApiPercent}%
                  </TableCell>
                  <TableCell>
                    <Chip
                      label={plan.available ? "Active" : "Inactive"}
                      color={plan.available ? "success" : "error"}
                      size="small"
                    />
                  </TableCell>
                  <TableCell align="right">
                    <IconButton
                      onClick={(e) => handleMenuClick(e, plan)}
                      size="small"
                    >
                      <MoreVertIcon />
                    </IconButton>
                    <Menu
                      anchorEl={anchorEl}
                      open={Boolean(anchorEl) && selectedPlan === plan}
                      onClose={handleMenuClose}
                    >
                      <MenuItem
                        onClick={() => {
                          handleFormOpen(plan);
                          handleMenuClose();
                        }}
                      >
                        Edit
                      </MenuItem>
                      <MenuItem
                        onClick={() => {
                          handleDelete(plan);
                          handleMenuClose();
                        }}
                      >
                        Delete
                      </MenuItem>
                    </Menu>
                  </TableCell>
                </TableRow>
              ))}
            </TableBody>
          </Table>
        </TableContainer>
      </Paper>

      {isFormOpen && (
        <AirtimeForm
          initialPlan={selectedPlan ?? undefined}
          onSubmit={handleFormSubmit}
          onClose={handleFormClose}
          isAddPlan={isAddPlan}
        />
      )}
    </Box>
  );
};

export default AirtimePage;
