import { FC, useState } from "react";
import Image from "next/image";
import {
  Card,
  CardContent,
  CardHeader,
  Avatar,
  IconButton,
  Menu,
  MenuItem,
  Typography,
  Chip,
  Button,
  Box,
  Divider,
  useTheme,
  useMediaQuery,
} from "@mui/material";
import {
  MoreVert,
  AccountCircle,
  AttachMoney,
  DateRange,
  CheckCircle,
  Cancel,
  Delete,
} from "@mui/icons-material";

interface UserCardProps {
  user: {
    _id: string;
    name: string;
    username: string;
    email: string;
    mobileNumber: string;
    image: string;
    accountBalance: number;
    dateJoined: Date;
    lastLogin: Date | null;
    refererBonus: number;
    transactionStatus: boolean;
    userType: string;
  };
  onViewProfile: (userId: string) => void;
  onDelete: (userId: string) => void;
}

const UserCard: FC<UserCardProps> = ({ user, onViewProfile, onDelete }) => {
  const [anchorEl, setAnchorEl] = useState<null | HTMLElement>(null);
  const theme = useTheme();
  const isSmallScreen = useMediaQuery(theme.breakpoints.down("sm"));
  const open = Boolean(anchorEl);

  const isUrl = (str: string) => {
    if (typeof str !== "string") return false;
    try {
      new URL(str);
      return true;
    } catch (_) {
      return false;
    }
  };

  const imageSrc = isUrl(user.image)
    ? user.image
    : user.image
    ? `/uploads/${user.image}`
    : "/default-avatar.png";

  const handleMenuClick = (event: React.MouseEvent<HTMLElement>) => {
    event.stopPropagation();
    setAnchorEl(event.currentTarget);
  };

  const handleMenuClose = () => {
    setAnchorEl(null);
  };

  const handleDelete = () => {
    onDelete(user._id);
    handleMenuClose();
  };

  return (
    <Card
      sx={{
        maxWidth: 345,
        minWidth: isSmallScreen ? "100%" : 300,
        m: 1,
        transition: "transform 0.3s, box-shadow 0.3s",
        "&:hover": {
          transform: "translateY(-5px)",
          boxShadow: theme.shadows[6],
        },
      }}
    >
      <CardHeader
        avatar={
          <Avatar
            sx={{
              width: 56,
              height: 56,
              border: `3px solid ${theme.palette.background.paper}`,
            }}
          >
            <Image
              src={imageSrc}
              alt={user.name}
              fill
              className="rounded-full"
              quality={80}
              loading="lazy"
              style={{ objectFit: "cover" }}
              sizes="56px"
            />
          </Avatar>
        }
        action={
          <>
            <IconButton
              aria-label="settings"
              onClick={handleMenuClick}
              sx={{ mt: 1 }}
            >
              <MoreVert />
            </IconButton>
            <Menu
              anchorEl={anchorEl}
              open={open}
              onClose={handleMenuClose}
              onClick={(e) => e.stopPropagation()}
            >
              <MenuItem onClick={handleDelete} sx={{ color: "error.main" }}>
                <Delete sx={{ mr: 1 }} /> Delete User
              </MenuItem>
            </Menu>
          </>
        }
        title={
          <Typography variant="h6" component="div" noWrap>
            {user.name}
          </Typography>
        }
        subheader={
          <Typography variant="body2" color="text.secondary" noWrap>
            @{user.username}
          </Typography>
        }
        sx={{ pb: 0 }}
      />

      <CardContent>
        <Box
          sx={{
            display: "flex",
            flexDirection: "column",
            gap: 1.5,
            mb: 2,
          }}
        >
          <Box sx={{ display: "flex", alignItems: "center" }}>
            <AttachMoney
              sx={{ color: "text.secondary", mr: 1, fontSize: "1.2rem" }}
            />
            <Typography variant="body2" color="text.secondary">
              Balance:
            </Typography>
            <Typography sx={{ ml: "auto", fontWeight: 500 }}>
              ₦{user.accountBalance?.toLocaleString() || "0"}
            </Typography>
          </Box>

          <Box sx={{ display: "flex", alignItems: "center" }}>
            <DateRange
              sx={{ color: "text.secondary", mr: 1, fontSize: "1.2rem" }}
            />
            <Typography variant="body2" color="text.secondary">
              Joined:
            </Typography>
            <Typography sx={{ ml: "auto" }}>
              {new Date(user.dateJoined).toLocaleDateString()}
            </Typography>
          </Box>

          <Box sx={{ display: "flex", alignItems: "center" }}>
            <AccountCircle
              sx={{ color: "text.secondary", mr: 1, fontSize: "1.2rem" }}
            />
            <Typography variant="body2" color="text.secondary">
              Type:
            </Typography>
            <Chip
              label={user.userType.toLowerCase()}
              size="small"
              sx={{
                ml: "auto",
                textTransform: "capitalize",
                bgcolor: "action.selected",
              }}
            />
          </Box>

          <Box sx={{ display: "flex", alignItems: "center" }}>
            {user.transactionStatus ? (
              <CheckCircle
                sx={{ color: "success.main", mr: 1, fontSize: "1.2rem" }}
              />
            ) : (
              <Cancel sx={{ color: "error.main", mr: 1, fontSize: "1.2rem" }} />
            )}
            <Typography variant="body2" color="text.secondary">
              Status:
            </Typography>
            <Chip
              label={user.transactionStatus ? "Active" : "Inactive"}
              size="small"
              sx={{
                ml: "auto",
                bgcolor: user.transactionStatus
                  ? "success.light"
                  : "error.light",
                color: user.transactionStatus ? "success.dark" : "error.dark",
              }}
            />
          </Box>
        </Box>

        <Divider sx={{ my: 1 }} />

        <Button
          fullWidth
          variant="contained"
          onClick={() => onViewProfile(user._id)}
          sx={{ mt: 2 }}
        >
          View Profile
        </Button>
      </CardContent>
    </Card>
  );
};

export default UserCard;
