"use client";

import React, { useState, useEffect } from "react";
import {
  Typography,
  Container,
  Grid,
  TextField,
  Switch,
  FormControlLabel,
  Button,
  ToggleButton,
  Box,
  Snackbar,
  Alert,
  Backdrop,
  Card,
  CardContent,
  useTheme,
  useMediaQuery,
  Radio,
  RadioGroup,
  FormControl,
  MenuItem,
  Select,
  InputLabel,
} from "@mui/material";
import Image from "next/image";
import axios from "axios";
import { useSession } from "next-auth/react";
import { AuthModal, ConfirmModal } from "@/app/components/modals/authmodal";
import TransactionModal from "@/app/components/modals/TransactionModal";
import mtnLogo from "../../../public/images/mtn.png";
import airtelLogo from "../../../public/images/aritel.png";
import mobile9Logo from "../../../public/images/9mobile.png";
import LoadingComponent from "@/app/components/loadingcomponent";
import gloLogo from "../../../public/images/glo.jpg";

type Variation = {
  variation_code: string;
  name: string;
  variation_amount: string;
  fixedPrice: string;
};

type ApiResponse = {
  response_description: string;
  content: {
    ServiceName: string;
    serviceID: string;
    variations: Variation[];
  };
};

const thirdPartyPlans: Record<string, Variation[]> = {
  "mtn-data": [],
  "airtel-data": [],
  "glo-data": [],
  "etisalat-data": [],
};

const fetchThirdPartyPlans = async (serviceID: string) => {
  try {
    const response = await axios.get<ApiResponse>(
      `https://vtpass.com/api/service-variations?serviceID=${serviceID}`
    );

    if (response.data && response.data.content.variations) {
      thirdPartyPlans[serviceID] = response.data.content.variations;
    }
  } catch (error) {
    console.error(`Error fetching ${serviceID} plans:`, error);
  }
};

async function extractPlanDetails(planString: string) {
  const match = planString.match(/(\d+(?:GB|MB))\s+N(\d+)/);
  if (match) {
    return { plansize: match[1], amount: parseFloat(match[2]) };
  }
  return { plansize: null, amount: 0 };
}

interface Plan {
  label: string;
  apiDetails: any;
  description?: string;
}

const BuyData: React.FC = () => {
  const { data: session } = useSession();
  const theme = useTheme();
  const isMobile = useMediaQuery(theme.breakpoints.down("sm"));
  const [plans, setPlans] = useState<Record<string, Record<string, Plan[]>>>(
    {}
  );
  const [userType, setUserType] = useState<string | null>(null);
  const [selectedNetwork, setSelectedNetwork] = useState<string | null>(null);
  const [selectedPlanType, setSelectedPlanType] = useState<string | null>(null);
  const [selectedPlan, setSelectedPlan] = useState<Plan | null>(null);
  const [mobileNumber, setMobileNumber] = useState<string>("");
  const [bypass, setBypass] = useState<boolean>(false);
  const [userId, setUserId] = useState<string | null>(null);
  const [userEmail, setUserEmail] = useState<string | null>(null);
  const [confirmOpen, setConfirmOpen] = useState(false);
  const [authOpen, setAuthOpen] = useState(false);
  const [isTransactionModalOpen, setIsTransactionModalOpen] = useState(false);
  const [isLoading, setIsLoading] = useState(false);
  const [isBackdropOpen, setIsBackdropOpen] = useState(false);
  const [snackbarOpen, setSnackbarOpen] = useState(false);
  const [snackbarMessage, setSnackbarMessage] = useState("");
  const [transactionPinStatus, setTransactionPinStatus] =
    useState<boolean>(false);
  const [thumbPrintStatus, setThumbPrintStatus] = useState<boolean>(false);
  const [walletBalance, setWalletBalance] = useState<string>("");
  const [dataSource, setDataSource] = useState<"proprietary" | "thirdParty">(
    "proprietary"
  );
  const [thirdPartyNetwork, setThirdPartyNetwork] = useState<string>("");
  const [thirdPartyPlanType, setThirdPartyPlanType] = useState<string>("all");
  const [selectedThirdPartyPlan, setSelectedThirdPartyPlan] =
    useState<Variation | null>(null);
  const [transactionDetails, setTransactionDetails] = useState({
    finalMessage: "",
    amount: "",
    recipient: "",
    date: "",
    status: "success" as "success" | "pending" | "failed",
  });

  useEffect(() => {
    const fetchUserType = async () => {
      try {
        setIsLoading(true);
        setIsBackdropOpen(true);
        const response = await axios.get(`/api/users/getSingleUser`);
        setUserType(response.data.userType);
        setUserId(response.data._id);
        setUserEmail(response.data.email);
        setTransactionPinStatus(response.data.transactionStatus);
        setWalletBalance(response.data.accountBalance);
        setThumbPrintStatus(response.data.thumbprintStatus);
      } catch (error) {
        console.error("Error fetching user type:", error);
      } finally {
        setIsLoading(false);
        setIsBackdropOpen(false);
      }
    };

    fetchUserType();
  }, [session]);

  useEffect(() => {
    const fetchDataPlans = async () => {
      try {
        if (!userType) return;
        const response = await axios.get("/api/adminapi/data");
        const dataPlans = response.data;
        const newPlans: Record<string, Record<string, Plan[]>> = {};

        dataPlans.forEach((plan: any) => {
          const {
            network,
            planType,
            planSize,
            planVolume,
            apiDetails,
            smartEarnerPrice,
            affiliatePrice,
            topUserPrice,
            description,
          } = plan;
          let planPrice;
          if (userType === "Smart Earner") {
            planPrice = smartEarnerPrice;
          } else if (userType === "Affiliate User") {
            planPrice = affiliatePrice;
          } else if (userType === "Top User") {
            planPrice = topUserPrice;
          }
          const size =
            planSize < 1 ? `${planSize * 1000}MB` : `${planSize}${planVolume}`;
          const planLabel = `${size} N${planPrice} - ${description}`;

          if (!newPlans[network]) {
            newPlans[network] = {};
          }
          if (!newPlans[network][planType]) {
            newPlans[network][planType] = [];
          }
          newPlans[network][planType].push({
            label: planLabel,
            apiDetails,
            description,
          });
        });
        setPlans(newPlans);
      } catch (error) {
        console.error("Error fetching data plans:", error);
      }
    };

    fetchDataPlans();
  }, [userType]);

  useEffect(() => {
    if (dataSource === "thirdParty" && thirdPartyNetwork) {
      const serviceID = `${thirdPartyNetwork.toLowerCase()}-data`;
      fetchThirdPartyPlans(serviceID);
    }
  }, [dataSource, thirdPartyNetwork]);

  const handleNetworkSelect = (network: string) => {
    setSelectedNetwork(network);
    setSelectedPlanType(null);
    setSelectedPlan(null);
  };

  const handlePlanTypeSelect = (planType: string) => {
    setSelectedPlanType(planType);
    setSelectedPlan(null);
  };

  const handlePlanSelect = (plan: Plan) => {
    setSelectedPlan(plan);
  };

  const handleThirdPartyPlanSelect = (plan: Variation) => {
    setSelectedThirdPartyPlan(plan);
  };

  const filteredThirdPartyPlans = () => {
    if (!thirdPartyNetwork) return [];
    const serviceID = `${thirdPartyNetwork.toLowerCase()}-data`;
    const allPlans = thirdPartyPlans[serviceID] || [];

    if (thirdPartyPlanType === "all") return allPlans;

    return allPlans.filter((plan) => {
      const planName = plan.name.toLowerCase();
      if (thirdPartyPlanType === "daily") {
        return planName.includes("day") || planName.includes("24 hrs");
      } else if (thirdPartyPlanType === "weekly") {
        return planName.includes("week") || planName.includes("7 days");
      } else if (thirdPartyPlanType === "monthly") {
        return planName.includes("month") || planName.includes("30 days");
      }
      return true;
    });
  };

  const handleSubmit = async () => {
    if (dataSource === "proprietary") {
      if (
        !selectedNetwork ||
        !selectedPlanType ||
        !selectedPlan ||
        !mobileNumber
      ) {
        setSnackbarMessage("Please fill in all required fields.");
        setSnackbarOpen(true);
        return;
      }
      let result = await extractPlanDetails(selectedPlan.label);
      let amount = result.amount;
      if (parseFloat(walletBalance) >= amount) {
        setTransactionDetails({
          finalMessage: `Are you sure you want to buy ${selectedPlan.label} ${selectedNetwork}'s ${selectedPlanType} for ${mobileNumber}?`,
          amount: amount.toString(),
          recipient: selectedNetwork,
          date: new Date().toLocaleString(),
          status: "success",
        });
        setConfirmOpen(true);
      } else {
        setSnackbarMessage("Insufficient Balance, Please Fund your wallet");
        setSnackbarOpen(true);
        setConfirmOpen(false);
      }
    } else {
      // Third party submission
      if (!thirdPartyNetwork || !selectedThirdPartyPlan || !mobileNumber) {
        setSnackbarMessage("Please fill in all required fields.");
        setSnackbarOpen(true);
        return;
      }
      const amount = parseFloat(selectedThirdPartyPlan.variation_amount);
      if (parseFloat(walletBalance) >= amount) {
        setTransactionDetails({
          finalMessage: `Are you sure you want to buy ${selectedThirdPartyPlan.name} for ${mobileNumber}?`,
          amount: selectedThirdPartyPlan.variation_amount,
          recipient: thirdPartyNetwork,
          date: new Date().toLocaleString(),
          status: "success",
        });
        setConfirmOpen(true);
      } else {
        setSnackbarMessage("Insufficient Balance, Please Fund your wallet");
        setSnackbarOpen(true);
        setConfirmOpen(false);
      }
    }
  };

  const handleConfirm = () => {
    setConfirmOpen(false);
    if (thumbPrintStatus || transactionPinStatus) {
      setAuthOpen(true);
    } else {
      handleVerify();
    }
  };

  const handleVerify = async () => {
    setAuthOpen(false);
    setIsLoading(true);
    setIsBackdropOpen(true);

    try {
      if (dataSource === "proprietary") {
        const serviceID = selectedNetwork
          ? `${selectedNetwork.toLowerCase()}-data`
          : "";
        const datasub = {
          network: selectedNetwork,
          planType: selectedPlanType,
          apiDetails: selectedPlan?.apiDetails ?? {},
          mobileNumber,
          bypass,
          plan: selectedPlan?.label,
          userId,
          userEmail,
          serviceID,
        };
        const response = await axios.post(
          "/api/buydata",
          JSON.stringify({ data: datasub }),
          {
            headers: { "Content-Type": "application/json" },
          }
        );
        if (response.status === 429) {
          setTransactionDetails({
            finalMessage: response.data.message,
            amount: selectedPlan?.label || "",
            recipient: mobileNumber,
            date: new Date().toLocaleString(),
            status: "failed",
          });
        } else if (response.data.status === true) {
          setTransactionDetails({
            finalMessage: `Transaction Successful! ${response.data.api_response}`,
            amount: selectedPlan?.label || "",
            recipient: mobileNumber,
            date: new Date().toLocaleString(),
            status: "success",
          });
        } else {
          setTransactionDetails({
            finalMessage: `Transaction Failed: ${response.data.message}`,
            amount: selectedPlan?.label || "",
            recipient: mobileNumber,
            date: new Date().toLocaleString(),
            status: "failed",
          });
        }
      } else {
        // Third party API call
        const response = await axios.post(
          "/api/vtpassdatasub",
          {
            network: thirdPartyNetwork,
            variation_code: selectedThirdPartyPlan?.variation_code,
            phone: mobileNumber,
            amount: selectedThirdPartyPlan?.variation_amount,
            userId,
            userEmail,
          },
          {
            headers: { "Content-Type": "application/json" },
          }
        );

        if (response.data.code === "000") {
          setTransactionDetails({
            finalMessage: `Transaction Successful! ${response.data.content.transactions.product_name}`,
            amount: selectedThirdPartyPlan?.variation_amount || "",
            recipient: mobileNumber,
            date: new Date().toLocaleString(),
            status: "success",
          });
        } else {
          setTransactionDetails({
            finalMessage: `Transaction Failed: ${response.data.response_description}`,
            amount: selectedThirdPartyPlan?.variation_amount || "",
            recipient: mobileNumber,
            date: new Date().toLocaleString(),
            status: "failed",
          });
        }
      }
      setIsTransactionModalOpen(true);
    } catch (error) {
      console.error(error);
      setTransactionDetails({
        finalMessage: "Transaction processing error. Please try again.",
        amount:
          dataSource === "proprietary"
            ? selectedPlan?.label || ""
            : selectedThirdPartyPlan?.variation_amount || "",
        recipient: mobileNumber,
        date: new Date().toLocaleString(),
        status: "failed",
      });
      setIsTransactionModalOpen(true);
    } finally {
      setIsLoading(false);
      setIsBackdropOpen(false);
    }
  };

  const handleTransactionModalClose = () => {
    setIsTransactionModalOpen(false);
    window.location.reload();
  };

  return (
    <Container maxWidth="md" sx={{ py: 3 }}>
      {/* Backdrop for Loading */}
      <Backdrop
        open={isBackdropOpen}
        sx={{ color: "#fff", zIndex: (theme) => theme.zIndex.drawer + 1 }}
      >
        <LoadingComponent />
      </Backdrop>

      <Card sx={{ mb: 3, boxShadow: 3 }}>
        <CardContent>
          <Typography variant="h4" gutterBottom>
            Buy Data
          </Typography>
          <Typography variant="body1" color="text.secondary">
            Select your preferred network and data plan
          </Typography>
        </CardContent>
      </Card>

      {/* Data Source Selection */}
      <Card sx={{ mb: 3, boxShadow: 2 }}>
        <CardContent>
          <Typography variant="h6" gutterBottom>
            Data Source
          </Typography>
          <FormControl component="fieldset">
            <RadioGroup
              row
              value={dataSource}
              onChange={(e) =>
                setDataSource(e.target.value as "proprietary" | "thirdParty")
              }
            >
              <FormControlLabel
                value="proprietary"
                control={<Radio />}
                label="Proprietary"
              />
              <FormControlLabel
                value="thirdParty"
                control={<Radio />}
                label="Third Party (VTPass)"
              />
            </RadioGroup>
          </FormControl>
        </CardContent>
      </Card>

      {dataSource === "proprietary" ? (
        <>
          {/* Network Selection */}
          <Card sx={{ mb: 3, boxShadow: 2 }}>
            <CardContent>
              <Typography variant="h6" gutterBottom>
                Select Network
              </Typography>
              <Grid container spacing={2}>
                {Object.keys(plans).map((network) => (
                  <Grid size={{ xs: 6, sm: 3 }} key={network}>
                    <ToggleButton
                      value={network}
                      selected={selectedNetwork === network}
                      onChange={() => handleNetworkSelect(network)}
                      sx={{
                        width: "100%",
                        height: isMobile ? "80px" : "100px",
                        p: 1,
                        border: `1px solid ${theme.palette.divider}`,
                        "&.Mui-selected": {
                          backgroundColor: theme.palette.action.selected,
                          borderColor: theme.palette.primary.main,
                        },
                      }}
                    >
                      <Box
                        sx={{
                          position: "relative",
                          width: "100%",
                          height: "100%",
                        }}
                      >
                        <Image
                          src={
                            network === "MTN"
                              ? mtnLogo
                              : network === "Airtel"
                              ? airtelLogo
                              : network === "Glo"
                              ? gloLogo
                              : mobile9Logo
                          }
                          alt={network}
                          fill
                          sizes="100%"
                          priority
                          style={{ objectFit: "contain" }}
                        />
                      </Box>
                    </ToggleButton>
                  </Grid>
                ))}
              </Grid>
            </CardContent>
          </Card>

          {selectedNetwork && (
            <Card sx={{ mb: 3, boxShadow: 2 }}>
              <CardContent>
                <Typography variant="h6" gutterBottom>
                  Select Plan Type
                </Typography>
                <Grid container spacing={2}>
                  {Object.keys(plans[selectedNetwork]).map((planType) => (
                    <Grid size={{ xs: 6, sm: 4, md: 3 }} key={planType}>
                      <ToggleButton
                        value={planType}
                        selected={selectedPlanType === planType}
                        onChange={() => handlePlanTypeSelect(planType)}
                        sx={{
                          width: "100%",
                          height: "45px",
                          textTransform: "none",
                          "&.Mui-selected": {
                            backgroundColor: theme.palette.action.selected,
                          },
                        }}
                      >
                        {planType}
                      </ToggleButton>
                    </Grid>
                  ))}
                </Grid>
              </CardContent>
            </Card>
          )}

          {selectedNetwork && selectedPlanType && (
            <Card sx={{ mb: 3, boxShadow: 2 }}>
              <CardContent>
                <Typography variant="h6" gutterBottom>
                  Select Data Plan
                </Typography>
                <Grid container spacing={2}>
                  {plans[selectedNetwork][selectedPlanType].map(
                    (plan, index) => (
                      <Grid size={{ xs: 6, sm: 4, md: 3 }} key={index}>
                        <ToggleButton
                          value={plan.label}
                          selected={selectedPlan?.label === plan.label}
                          onChange={() => handlePlanSelect(plan)}
                          sx={{
                            fontSize: "60%",
                            width: "105%",
                            height: "60px",
                            fontWeight: "600",
                            whiteSpace: "wrap",
                            overflow: "hidden",
                            textOverflow: "ellipsis",
                            "&.Mui-selected": {
                              backgroundColor: theme.palette.action.selected,
                            },
                          }}
                        >
                          {plan.label}
                        </ToggleButton>
                      </Grid>
                    )
                  )}
                </Grid>
              </CardContent>
            </Card>
          )}
        </>
      ) : (
        <>
          {/* Third Party Network Selection */}
          <Card sx={{ mb: 3, boxShadow: 2 }}>
            <CardContent>
              <Typography variant="h6" gutterBottom>
                Select Network
              </Typography>
              <Grid container spacing={2}>
                {["MTN", "Airtel", "Glo", "9mobile"].map((network) => (
                  <Grid size={{ xs: 6, sm: 3 }} key={network}>
                    <ToggleButton
                      value={network}
                      selected={thirdPartyNetwork === network}
                      onChange={() => setThirdPartyNetwork(network)}
                      sx={{
                        width: "100%",
                        height: isMobile ? "80px" : "100px",
                        p: 1,
                        border: `1px solid ${theme.palette.divider}`,
                        "&.Mui-selected": {
                          backgroundColor: theme.palette.action.selected,
                          borderColor: theme.palette.primary.main,
                        },
                      }}
                    >
                      <Box
                        sx={{
                          position: "relative",
                          width: "100%",
                          height: "100%",
                        }}
                      >
                        <Image
                          src={
                            network === "MTN"
                              ? mtnLogo
                              : network === "Airtel"
                              ? airtelLogo
                              : network === "Glo"
                              ? gloLogo
                              : mobile9Logo
                          }
                          alt={network}
                          fill
                          sizes="100%"
                          priority
                          style={{ objectFit: "contain" }}
                        />
                      </Box>
                    </ToggleButton>
                  </Grid>
                ))}
              </Grid>
            </CardContent>
          </Card>

          {thirdPartyNetwork && (
            <>
              <Card sx={{ mb: 3, boxShadow: 2 }}>
                <CardContent>
                  <Typography variant="h6" gutterBottom>
                    Select Plan Type
                  </Typography>
                  <FormControl fullWidth>
                    <InputLabel>Plan Duration</InputLabel>
                    <Select
                      value={thirdPartyPlanType}
                      onChange={(e) => setThirdPartyPlanType(e.target.value)}
                      label="Plan Duration"
                    >
                      <MenuItem value="all">All Plans</MenuItem>
                      <MenuItem value="daily">Daily</MenuItem>
                      <MenuItem value="weekly">Weekly</MenuItem>
                      <MenuItem value="monthly">Monthly</MenuItem>
                    </Select>
                  </FormControl>
                </CardContent>
              </Card>

              <Card sx={{ mb: 3, boxShadow: 2 }}>
                <CardContent>
                  <Typography variant="h6" gutterBottom>
                    Select Data Plan
                  </Typography>
                  <Grid container spacing={2}>
                    {filteredThirdPartyPlans().map((plan, index) => (
                      <Grid size={{ xs: 12 }} key={index}>
                        <ToggleButton
                          value={plan.variation_code}
                          selected={
                            selectedThirdPartyPlan?.variation_code ===
                            plan.variation_code
                          }
                          onChange={() => handleThirdPartyPlanSelect(plan)}
                          sx={{
                            width: "100%",
                            minHeight: "60px",
                            textAlign: "left",
                            justifyContent: "flex-start",
                            "&.Mui-selected": {
                              backgroundColor: theme.palette.action.selected,
                            },
                          }}
                        >
                          <Box>
                            <Typography fontWeight="bold">
                              {plan.name}
                            </Typography>
                            <Typography>₦{plan.variation_amount}</Typography>
                          </Box>
                        </ToggleButton>
                      </Grid>
                    ))}
                  </Grid>
                </CardContent>
              </Card>
            </>
          )}
        </>
      )}

      {(selectedNetwork || thirdPartyNetwork) &&
        ((dataSource === "proprietary" && selectedPlan) ||
          (dataSource === "thirdParty" && selectedThirdPartyPlan)) && (
          <Card sx={{ mb: 3, boxShadow: 2 }}>
            <CardContent>
              <Typography variant="h6" gutterBottom>
                Enter Details
              </Typography>
              <TextField
                fullWidth
                label="Phone Number"
                placeholder="Enter phone number"
                value={mobileNumber}
                onChange={(e) => setMobileNumber(e.target.value)}
                sx={{ mb: 2 }}
              />
              {dataSource === "proprietary" && (
                <FormControlLabel
                  control={
                    <Switch
                      checked={bypass}
                      onChange={(e) => setBypass(e.target.checked)}
                    />
                  }
                  label="Bypass Phone Number"
                  sx={{ mb: 2 }}
                />
              )}
              <Button
                variant="contained"
                color="primary"
                fullWidth
                size="large"
                onClick={handleSubmit}
                disabled={isLoading}
              >
                {isLoading ? "Processing..." : "Buy Data"}
              </Button>
            </CardContent>
          </Card>
        )}

      <ConfirmModal
        open={confirmOpen}
        onClose={() => setConfirmOpen(false)}
        onConfirm={handleConfirm}
        transactionDetails={{
          message: transactionDetails.finalMessage,
          amount: transactionDetails.amount,
          recipient: mobileNumber,
          date: new Date().toLocaleDateString(),
        }}
      />
      <AuthModal
        open={authOpen}
        onClose={() => setAuthOpen(false)}
        onVerify={handleVerify}
      />
      <TransactionModal
        open={isTransactionModalOpen}
        onClose={handleTransactionModalClose}
        transactionDetails={transactionDetails}
        isLoading={isLoading}
      />
      <Snackbar
        open={snackbarOpen}
        autoHideDuration={6000}
        onClose={() => setSnackbarOpen(false)}
        anchorOrigin={{ vertical: "top", horizontal: "center" }}
      >
        <Alert
          onClose={() => setSnackbarOpen(false)}
          severity={
            transactionDetails.status === "success" ? "success" : "error"
          }
          sx={{ width: "100%" }}
        >
          {snackbarMessage}
        </Alert>
      </Snackbar>
    </Container>
  );
};

export default BuyData;
