"use client";
import { useEffect, useState } from "react";
import {
  Box,
  Typography,
  Card,
  CardContent,
  Avatar,
  Grid,
  Paper,
  Table,
  TableBody,
  TableCell,
  TableContainer,
  TableHead,
  TableRow,
  useTheme,
  useMediaQuery,
  Chip,
} from "@mui/material";
import { format } from "date-fns";
import LoadingComponent from "@/app/components/loadingcomponent";

interface ReferralDetail {
  email: string;
  name: string;
  image: string;
  lastLogin: string | Date;
}

interface ReferralStats {
  totalReferrals: number;
  totalDeposits: number;
}

const Referrals = () => {
  const theme = useTheme();
  const isMobile = useMediaQuery(theme.breakpoints.down("sm"));
  const [referralDetails, setReferralDetails] = useState<ReferralDetail[]>([]);
  const [stats, setStats] = useState<ReferralStats | null>(null);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState<string | null>(null);

  useEffect(() => {
    const fetchData = async () => {
      try {
        setLoading(true);
        const response = await fetch("/api/referral/details");
        const data = await response.json();

        if (response.ok) {
          setReferralDetails(data.referralDetails || []);
          setStats({
            totalReferrals: data.totalReferrals || 0,
            totalDeposits: data.totalDeposits || 0,
          });
        } else {
          setError(data.error || "Failed to fetch referral details");
        }
      } catch (err) {
        setError("Network error. Please try again later.");
        console.error("Failed to fetch referral details:", err);
      } finally {
        setLoading(false);
      }
    };

    fetchData();
  }, []);

  const formatDate = (dateString: string | Date) => {
    if (!dateString) return "Never logged in";
    try {
      const date = new Date(dateString);
      return format(date, "MMM dd, yyyy hh:mm a");
    } catch {
      return "Invalid date";
    }
  };

  if (loading) {
    return (
      <Box
        display="flex"
        justifyContent="center"
        alignItems="center"
        minHeight="200px"
      >
        <LoadingComponent />
      </Box>
    );
  }

  if (error) {
    return (
      <Box textAlign="center" p={4}>
        <Typography color="error">{error}</Typography>
      </Box>
    );
  }

  return (
    <Box sx={{ p: isMobile ? 1 : 3 }}>
      <Typography variant="h4" gutterBottom sx={{ mb: 3 }}>
        Your Referrals
      </Typography>

      {/* Stats Cards */}
      <Grid container spacing={3} sx={{ mb: 4 }}>
        <Grid size={{ xs: 12, sm: 6, md: 4 }}>
          <Card elevation={3}>
            <CardContent>
              <Typography variant="h6" color="text.secondary">
                Total Referrals
              </Typography>
              <Typography variant="h3" sx={{ fontWeight: "bold" }}>
                {stats?.totalReferrals || 0}
              </Typography>
            </CardContent>
          </Card>
        </Grid>
        <Grid size={{ xs: 12, sm: 6, md: 4 }}>
          <Card elevation={3}>
            <CardContent>
              <Typography variant="h6" color="text.secondary">
                Total Earnings
              </Typography>
              <Typography variant="h3" sx={{ fontWeight: "bold" }}>
                ₦{(stats?.totalDeposits || 0).toLocaleString()}
              </Typography>
            </CardContent>
          </Card>
        </Grid>
        <Grid size={{ xs: 12, md: 4 }}>
          <Card elevation={3}>
            <CardContent>
              <Typography variant="h6" color="text.secondary">
                Active Referrals
              </Typography>
              <Typography variant="h3" sx={{ fontWeight: "bold" }}>
                {
                  referralDetails.filter(
                    (ref) => ref.lastLogin && ref.lastLogin !== "No login data"
                  ).length
                }
              </Typography>
            </CardContent>
          </Card>
        </Grid>
      </Grid>

      {/* Referrals Table */}
      <Card elevation={3}>
        <CardContent>
          <Typography variant="h6" gutterBottom>
            Referral Details
          </Typography>

          {referralDetails.length === 0 ? (
            <Box textAlign="center" py={4}>
              <Typography variant="body1" color="text.secondary">
                No referrals found. Share your referral link to invite friends!
              </Typography>
            </Box>
          ) : isMobile ? (
            <Grid container spacing={2}>
              {referralDetails.map((ref, index) => (
                <Grid size={{ xs: 12 }} key={index}>
                  <Paper elevation={1} sx={{ p: 2 }}>
                    <Box display="flex" alignItems="center" mb={1}>
                      <Avatar src={ref.image} sx={{ mr: 2 }}>
                        {ref.name?.charAt(0) || "U"}
                      </Avatar>
                      <Box>
                        <Typography fontWeight="bold">{ref.name}</Typography>
                        <Typography variant="body2" color="text.secondary">
                          {ref.email}
                        </Typography>
                      </Box>
                    </Box>
                    <Typography variant="body2">
                      Last Active: {formatDate(ref.lastLogin)}
                    </Typography>
                  </Paper>
                </Grid>
              ))}
            </Grid>
          ) : (
            <TableContainer component={Paper} elevation={0}>
              <Table>
                <TableHead>
                  <TableRow>
                    <TableCell>User</TableCell>
                    <TableCell>Email</TableCell>
                    <TableCell>Last Active</TableCell>
                    <TableCell align="right">Status</TableCell>
                  </TableRow>
                </TableHead>
                <TableBody>
                  {referralDetails.map((ref, index) => (
                    <TableRow key={index}>
                      <TableCell>
                        <Box display="flex" alignItems="center">
                          <Avatar src={ref.image} sx={{ mr: 2 }}>
                            {ref.name?.charAt(0) || "U"}
                          </Avatar>
                          <Typography>{ref.name}</Typography>
                        </Box>
                      </TableCell>
                      <TableCell>{ref.email}</TableCell>
                      <TableCell>{formatDate(ref.lastLogin)}</TableCell>
                      <TableCell align="right">
                        <Chip
                          label={
                            ref.lastLogin && ref.lastLogin !== "No login data"
                              ? "Active"
                              : "Inactive"
                          }
                          color={
                            ref.lastLogin && ref.lastLogin !== "No login data"
                              ? "success"
                              : "default"
                          }
                          size="small"
                        />
                      </TableCell>
                    </TableRow>
                  ))}
                </TableBody>
              </Table>
            </TableContainer>
          )}
        </CardContent>
      </Card>
    </Box>
  );
};

export default Referrals;
