"use client";

import React, { useEffect, useState } from "react";
import { useRouter } from "next/navigation";
import { useSession } from "next-auth/react";
import {
  Typography,
  Box,
  Container,
  Grid,
  Paper,
  Button,
  IconButton,
  Modal,
  Snackbar,
  Card,
  CardContent,
  useTheme,
  useMediaQuery,
} from "@mui/material";
import {
  FourGPlusMobiledata,
  PhoneInTalk,
  Groups,
  ReceiptLong,
  Payments,
  RequestQuote,
  Bolt,
  Tv,
  Sms,
  Person,
  AccountBalanceWallet,
  AttachMoney,
  PeopleAlt,
  Storage,
} from "@mui/icons-material";
import ContentCopyIcon from "@mui/icons-material/ContentCopy";
import KycForm from "../components/kyc";
import Image from "next/image";
import logo from "@/public/images/sabmuent.jpeg";
import NotificationModal from "../components/modals/NotificationModal";

export default function DashboardHome() {
  const router = useRouter();
  const { data: session, update } = useSession();
  const [username, setUsername] = useState("");
  const [openModal, setOpenModal] = useState(false);
  const [analytics, setAnalytics] = useState({
    walletBalance: 0,
    totalFunding: 0,
    referralBonus: 0,
    totalDataSold: 0,
  });
  const [referralLink, setReferralLink] = useState("");
  const [snackbarOpen, setSnackbarOpen] = useState(false);
  const [userEmail, setUserEmail] = useState("");
  const theme = useTheme();
  const isMobile = useMediaQuery(theme.breakpoints.down("sm"));

  // Fetch user info and analytics data
  useEffect(() => {
    const fetchUserInfo = async () => {
      const userName = session?.user?.name;
      const uEmail = session?.user?.email;
      if (userName) {
        setUserEmail(uEmail ?? "");
        setUsername(userName.split(" ")[0]); // Get only the first name
      }

      if (session?.user && !session.user.hasSeenModal) {
        setOpenModal(true);
      }
    };

    const fetchAnalytics = async () => {
      try {
        const response = await fetch("/api/analytics");
        const data = await response.json();
        setAnalytics(data);
      } catch (error) {
        console.error("Error fetching analytics:", error);
      }
    };

    const fetchReferralLink = async () => {
      try {
        const response = await fetch("/api/referral/create");
        const data = await response.json();
        if (data.referralLink) {
          setReferralLink(data.referralLink);
        }
      } catch (error) {
        console.error("Error fetching referral link:", error);
      }
    };

    fetchReferralLink();
    fetchUserInfo();
    fetchAnalytics();
  }, [session]);

  const handleCloseModal = async () => {
    setOpenModal(false);

    await fetch("/api/users/updateuser", {
      method: "PUT",
      headers: {
        "Content-Type": "application/json",
      },
      body: JSON.stringify({ hasSeenModal: true }),
    });

    update({ hasSeenModal: true });
  };

  const handleNavigation = (section: string) => {
    router.push(`/dashboard/${section}`);
  };

  const handleCopyReferralLink = () => {
    if (referralLink) {
      navigator.clipboard.writeText(referralLink);
      setSnackbarOpen(true);
    }
  };

  const handleCloseSnackbar = () => {
    setSnackbarOpen(false);
  };

  // Wallet metrics data
  const walletMetrics = [
    {
      label: "Wallet Balance",
      value: `₦${analytics.walletBalance.toLocaleString()}`,
      icon: <AccountBalanceWallet color="primary" />,
      color: theme.palette.primary.main,
    },
    {
      label: "Total Funding",
      value: `₦${analytics.totalFunding.toLocaleString()}`,
      icon: <AttachMoney color="success" />,
      color: theme.palette.success.main,
    },
    {
      label: "Referral Bonus",
      value: `₦${analytics.referralBonus.toLocaleString()}`,
      icon: <PeopleAlt color="secondary" />,
      color: theme.palette.secondary.main,
    },
    {
      label: "Total Data Sold",
      value: `${analytics.totalDataSold.toLocaleString()} GB`,
      icon: <Storage color="info" />,
      color: theme.palette.info.main,
    },
  ];

  // Dashboard services
  const services = [
    {
      label: "Data",
      icon: (
        <FourGPlusMobiledata
          fontSize={isMobile ? "medium" : "large"}
          color="primary"
        />
      ),
      section: "buydata",
    },
    {
      label: "Airtime",
      icon: (
        <PhoneInTalk fontSize={isMobile ? "medium" : "large"} color="primary" />
      ),
      section: "buyairtime",
    },
    {
      label: "Electricity",
      icon: <Bolt fontSize={isMobile ? "medium" : "large"} color="primary" />,
      section: "utilitypayments",
    },
    {
      label: "Cable Sub",
      icon: <Tv fontSize={isMobile ? "medium" : "large"} color="primary" />,
      section: "utilitypayments",
    },
    {
      label: "Bulk SMS",
      icon: <Sms fontSize={isMobile ? "medium" : "large"} color="primary" />,
      section: "buydata",
    },
    {
      label: "Referrals",
      icon: <Groups fontSize={isMobile ? "medium" : "large"} color="primary" />,
      section: "myreferrals",
    },
    {
      label: "Transactions",
      icon: (
        <ReceiptLong fontSize={isMobile ? "medium" : "large"} color="primary" />
      ),
      section: "transactions",
    },
    {
      label: "Pricing",
      icon: (
        <RequestQuote
          fontSize={isMobile ? "medium" : "large"}
          color="primary"
        />
      ),
      section: "pricing",
    },
    {
      label: "Profile",
      icon: <Person fontSize={isMobile ? "medium" : "large"} color="primary" />,
      section: "settings",
    },
    {
      label: "Fund Wallet",
      icon: (
        <Payments fontSize={isMobile ? "medium" : "large"} color="primary" />
      ),
      section: "fundwallet",
    },
  ];

  return (
    <Container maxWidth="lg" sx={{ py: 3 }}>
      {/* Welcome Section */}
      <Box
        sx={{
          display: "flex",
          justifyContent: "space-between",
          alignItems: "center",
          mb: 3,
        }}
      >
        <Box>
          <Typography variant="h5" component="h1">
            Welcome back, {username}!
          </Typography>
          <Typography variant="body1" color="text.secondary">
            Manage your services and track your transactions
          </Typography>
          <NotificationModal userEmail={userEmail} />
          <Button
            variant="contained"
            color="primary"
            size="small"
            sx={{ mt: 2 }}
            onClick={() => handleNavigation("fundwallet")}
          >
            Fund Wallet
          </Button>
        </Box>

        <Image
          src={logo}
          alt="SABAMUENT Logo"
          width={56}
          height={56}
          style={{ objectFit: "contain" }}
        />
      </Box>

      {/* Membership & Referral Section */}
      <Grid container spacing={2} sx={{ mb: 3 }}>
        <Grid size={{ xs: 12, sm: 6 }}>
          <Paper elevation={2} sx={{ p: 2, height: "100%" }}>
            <Typography variant="h6" gutterBottom>
              Refer & Earn
            </Typography>
            <Box display="flex" alignItems="center" flexWrap="wrap">
              <Typography variant="body2" sx={{ mr: 1 }}>
                Share your code:
              </Typography>
              <Box
                sx={{
                  display: "flex",
                  alignItems: "center",
                  backgroundColor: theme.palette.grey[100],
                  px: 1,
                  borderRadius: 1,
                  flexGrow: 1,
                }}
              >
                <Typography
                  variant="body2"
                  sx={{
                    fontFamily: "monospace",
                    overflow: "hidden",
                    textOverflow: "ellipsis",
                  }}
                >
                  {referralLink}
                </Typography>
                <IconButton onClick={handleCopyReferralLink} size="small">
                  <ContentCopyIcon fontSize="small" />
                </IconButton>
              </Box>
            </Box>
          </Paper>
        </Grid>
      </Grid>

      {/* Wallet Metrics Section */}
      <Grid container spacing={2} sx={{ mb: 3 }}>
        {walletMetrics.map((metric, index) => (
          <Grid size={{ xs: 6, sm: 6, md: 3 }} key={index}>
            <Card
              sx={{
                height: "100%",
                // w  idth: "138px",
                display: "flex",
                flexDirection: "column",
                boxShadow: 2,
                borderLeft: `4px solid ${metric.color}`,
              }}
            >
              <CardContent sx={{ flexGrow: 1 }}>
                <Box display="flex" alignItems="center" mb={1}>
                  {metric.icon}
                  <Typography
                    variant="subtitle2"
                    sx={{ ml: 1, color: "text.secondary" }}
                  >
                    {metric.label}
                  </Typography>
                </Box>
                <Typography variant="h5" sx={{ fontWeight: "bold" }}>
                  {metric.value}
                </Typography>
              </CardContent>
            </Card>
          </Grid>
        ))}
      </Grid>

      {/* Services Grid */}
      <Typography variant="h5" sx={{ mb: 2, mt: 4 }}>
        Our Services
      </Typography>
      <Grid container spacing={2}>
        {services.map((service, index) => (
          <Grid size={{ xs: 6, sm: 4, md: 3 }} key={index}>
            <Paper
              elevation={2}
              sx={{
                p: 2,
                textAlign: "center",
                cursor: "pointer",
                transition: "transform 0.2s, box-shadow 0.2s",
                "&:hover": {
                  transform: "translateY(-4px)",
                  boxShadow: 4,
                  backgroundColor: theme.palette.action.hover,
                },
                height: "100%",
                display: "flex",
                flexDirection: "column",
                justifyContent: "center",
              }}
              onClick={() => handleNavigation(service.section)}
            >
              <Box sx={{ mb: 1 }}>{service.icon}</Box>
              <Typography variant="subtitle1">{service.label}</Typography>
            </Paper>
          </Grid>
        ))}
      </Grid>

      {/* KYC Modal */}
      <Modal open={openModal} onClose={handleCloseModal}>
        <Box
          sx={{
            position: "absolute",
            top: "52%",
            left: "50%",
            transform: "translate(-50%, -50%)",
            width: isMobile ? "80%" : 400,
            bgcolor: "background.paper",
            borderRadius: 2,
            boxShadow: 24,
            outline: "none",
          }}
        >
          <Typography
            variant="body1"
            component="h4"
            gutterBottom
            sx={{ p: 2, textAlign: "center" }}
          >
            Welcome {username}!, please complete your KYC Verification
          </Typography>

          <KycForm handleCloseModal={handleCloseModal} />
          <Box sx={{ mt: 2, display: "flex", justifyContent: "flex-end" }}>
            <Button
              variant="outlined"
              onClick={handleCloseModal}
              sx={{ mr: 1, mb: 2 }}
            >
              Maybe Later
            </Button>
          </Box>
          <Typography
            variant="body1"
            component="h6"
            gutterBottom
            sx={{ textAlign: "center" }}
          >
            Please note we only use the information you provide to get you a
            virtual account and nothing more
          </Typography>
        </Box>
      </Modal>

      <Snackbar
        open={snackbarOpen}
        autoHideDuration={3000}
        onClose={handleCloseSnackbar}
        message="Referral link copied to clipboard!"
        anchorOrigin={{ vertical: "bottom", horizontal: "center" }}
      />
    </Container>
  );
}
