"use client";

import React, { useEffect, useState } from "react";
import {
  Button,
  TextField,
  Typography,
  Snackbar,
  Alert,
  Select,
  MenuItem,
  FormControl,
  InputLabel,
  Backdrop,
  Box,
  Card,
  CardContent,
  Grid,
  Divider,
  Paper,
  useMediaQuery,
  Theme,
} from "@mui/material";
import { AuthModal, ConfirmModal } from "../../components/modals/authmodal";
import TransactionModal from "@/app/components/modals/TransactionModal";
import axios from "axios";
import { useTheme } from "@mui/material/styles";
import { useSession } from "next-auth/react";
import LoadingComponent from "@/app/components/loadingcomponent";

const electricityProviders = [
  { value: "ikeja-electric", label: "Ikeja Electric" },
  { value: "eko-electric", label: "Eko Electric" },
  { value: "kano-electric", label: "Kano Electric" },
  { value: "portharcourt-electric", label: "Port Harcourt Electric" },
  { value: "jos-electric", label: "Jos Electric" },
  { value: "ibadan-electric", label: "Ibadan Electric" },
  { value: "kaduna-electric", label: "Kaduna Electric" },
  { value: "abuja-electric", label: "Abuja Electric" },
  { value: "enugu-electric", label: "Enugu Electric" },
  { value: "benin-electric", label: "Benin Electric" },
  { value: "aba-electric", label: "Aba Electric" },
  { value: "yola-Electric", label: "Yola Electric" },
];

const paymentTypes = ["Prepaid", "Postpaid"];

const BillPayment: React.FC = () => {
  const { data: session } = useSession();
  const [provider, setProvider] = useState("");
  const [paymentType, setPaymentType] = useState("");
  const [meterNumber, setMeterNumber] = useState("");
  const [customerName, setCustomerName] = useState("");
  const [customerAddress, setCustomerAddress] = useState("");
  const [meterType, setMeterType] = useState("");
  const [amount, setAmount] = useState("");
  const [customerNumber, setCustomerNumber] = useState("");
  const [isVerified, setIsVerified] = useState(false);
  const [isConfirmModalOpen, setIsConfirmModalOpen] = useState(false);
  const [isAuthModalOpen, setIsAuthModalOpen] = useState(false);
  const [isTransactionModalOpen, setIsTransactionModalOpen] = useState(false);
  const [isLoading, setIsLoading] = useState(false);
  const [isBackdropOpen, setIsBackdropOpen] = useState(false);
  const [snackbarOpen, setSnackbarOpen] = useState(false);
  const [transactionPinStatus, setTransactionPinStatus] =
    useState<boolean>(false);
  const [thumbPrintStatus, setThumbPrintStatus] = useState<boolean>(false);
  const [walletBalance, setWalletBalance] = useState<string>("");
  const [snackbarMessage, setSnackbarMessage] = useState("");
  const [transactionDetails, setTransactionDetails] = useState({
    finalMessage: "",
    amount: "",
    recipient: "",
    date: "",
    status: "success" as "success" | "pending" | "failed",
  });

  const theme = useTheme();
  const isMobile = useMediaQuery(theme.breakpoints.down("sm"));

  useEffect(() => {
    const fetchUserType = async () => {
      try {
        const response = await axios.get(`/api/users/getSingleUser`);
        setWalletBalance(response.data.accountBalance);
        setTransactionPinStatus(response.data.transactionStatus);
        setThumbPrintStatus(response.data.thumbprintStatus);
      } catch (error) {
        console.error("Error fetching user type:", error);
      }
    };

    fetchUserType();
  }, [session]);

  const handleVerify = async () => {
    if (!provider || !paymentType || !meterNumber) {
      setSnackbarMessage("Please fill in all required fields.");
      setSnackbarOpen(true);
      return;
    }

    setIsLoading(true);
    setIsBackdropOpen(true);
    try {
      const response = await fetch("/api/cablebillverification", {
        method: "POST",
        headers: { "Content-Type": "application/json" },
        body: JSON.stringify({ meterNumber, paymentType, provider }),
      });
      const data = await response.json();
      if (data.code === "000" && data.content.Customer_Name != "") {
        setCustomerName(data.content.Customer_Name);
        setCustomerAddress(data.content.Address);
        setMeterType(data.content.Meter_Type);
        setIsVerified(true);
        setSnackbarMessage("Verification successful.");
      } else {
        setIsVerified(false);
        setSnackbarMessage(`Verification failed: ${data.content.error}`);
      }
    } catch (error) {
      setSnackbarMessage("Verification Failed. Please try again.");
    } finally {
      setIsLoading(false);
      setIsBackdropOpen(false);
      setSnackbarOpen(true);
    }
  };

  const handleSubmit = () => {
    if (
      !provider ||
      !paymentType ||
      !meterNumber ||
      !amount ||
      !customerNumber
    ) {
      setSnackbarMessage("Please fill in all required fields.");
      setSnackbarOpen(true);
      return;
    }
    if (parseFloat(walletBalance) >= parseFloat(amount)) {
      setTransactionDetails({
        finalMessage: `Are you sure you want to pay ₦${amount} for ${customerName}'s ${paymentType} meter with Meter Number ${meterNumber}?`,
        amount,
        recipient: customerName,
        date: new Date().toLocaleString(),
        status: "success",
      });
      setIsConfirmModalOpen(true);
    } else {
      setSnackbarMessage("Insufficient Balance, Please Fund your Wallet");
      setSnackbarOpen(true);
      setIsConfirmModalOpen(false);
    }
  };

  const handleConfirm = () => {
    setIsConfirmModalOpen(false);
    if (thumbPrintStatus || transactionPinStatus) {
      setIsAuthModalOpen(true);
    } else {
      handleProcessPayment();
    }
  };

  const handleProcessPayment = async (pin?: string) => {
    setIsAuthModalOpen(false);
    setIsLoading(true);
    setIsBackdropOpen(true);
    try {
      const response = await fetch("/api/buyelectricity", {
        method: "POST",
        headers: { "Content-Type": "application/json" },
        body: JSON.stringify({
          meterNumber,
          paymentType,
          provider,
          amount,
          customerNumber,
        }),
      });
      const result = await response.json();

      setTransactionDetails((prev) => ({
        ...prev,
        finalMessage:
          result?.code === "000" &&
          result?.content?.transactions?.status === "delivered" &&
          paymentType === "Prepaid"
            ? `Payment of ₦${amount} for ${customerName}'s ${paymentType} meter was successful. The Prepaid token is ${result.purchased_code}`
            : result?.code === "000" &&
              result?.content?.transactions?.status === "delivered" &&
              paymentType === "Postpaid"
            ? `Payment of ₦${amount} for ${customerName}'s ${paymentType} meter was successful.`
            : result?.code === "000" &&
              result?.content?.transactions?.status === "pending"
            ? `Payment of ₦${amount} for ${customerName}'s ${paymentType} meter is pending. Please check back later.`
            : result?.code === "016" &&
              result?.content?.transactions?.status === "failed"
            ? `Payment failed due to an issue with the transaction. Please try again.`
            : result?.code === "018" ||
              result?.content?.transactions?.status === "failed"
            ? `Payment failed. Please try again.`
            : result?.code === "013" ||
              result?.response_description === "BELOW MINIMUM AMOUNT ALLOWED"
            ? `The Amount entered is below the minimum allowed. Please enter a valid amount.`
            : `Payment status unknown. Please contact support.`,
        status:
          result?.code === "000" &&
          result?.content?.transactions?.status === "delivered"
            ? "success"
            : result?.code === "000" &&
              result?.content?.transactions?.status === "pending"
            ? "pending"
            : result?.code === "016" &&
              result?.content?.transactions?.status === "failed"
            ? "failed"
            : result?.code === "018" ||
              result?.content?.transactions?.status === "failed"
            ? "failed"
            : result?.code === "013" ||
              result?.response_description === "BELOW MINIMUM AMOUNT ALLOWED"
            ? "failed"
            : "failed",
      }));

      setSnackbarMessage(
        result?.code === "000" &&
          result?.content?.transactions?.status === "delivered"
          ? "Payment successful."
          : "Payment failed."
      );
    } catch (error) {
      setTransactionDetails((prev) => ({
        ...prev,
        finalMessage: "Payment processing error. Please try again.",
        status: "failed",
      }));
      setSnackbarMessage("Payment processing error. Try again.");
    } finally {
      setIsTransactionModalOpen(true);
      setIsLoading(false);
      setIsBackdropOpen(false);
      setSnackbarOpen(true);
    }
  };

  const handleTransactionModalClose = () => {
    setIsTransactionModalOpen(false);
    window.location.reload();
  };

  return (
    <Box sx={{ maxWidth: 800, margin: "auto", p: isMobile ? 1 : 3 }}>
      {/* Backdrop for Loading */}
      <Backdrop
        open={isBackdropOpen}
        sx={{ color: "#fff", zIndex: (theme) => theme.zIndex.drawer + 1 }}
      >
        <LoadingComponent />
      </Backdrop>

      {/* Snackbar for Notifications */}
      <Snackbar
        open={snackbarOpen}
        autoHideDuration={3000}
        onClose={() => setSnackbarOpen(false)}
        anchorOrigin={{ vertical: "top", horizontal: "center" }}
      >
        <Alert
          severity={
            transactionDetails.status === "success" ? "success" : "error"
          }
          sx={{ width: "100%" }}
        >
          {snackbarMessage}
        </Alert>
      </Snackbar>

      <Card elevation={3} sx={{ mb: 3 }}>
        <CardContent>
          <Typography variant="h6" gutterBottom sx={{ fontWeight: "bold" }}>
            Electricity Bill Payment
          </Typography>
          <Divider sx={{ mb: 3 }} />

          <Grid container spacing={2}>
            <Grid size={{ xs: 12, sm: 6 }}>
              <FormControl fullWidth size="small" sx={{ mb: 2 }}>
                <InputLabel>Electricity Provider</InputLabel>
                <Select
                  value={provider}
                  onChange={(e) => setProvider(e.target.value)}
                  label="Electricity Provider"
                >
                  {electricityProviders.map((provider) => (
                    <MenuItem key={provider.value} value={provider.value}>
                      {provider.label}
                    </MenuItem>
                  ))}
                </Select>
              </FormControl>
            </Grid>

            <Grid size={{ xs: 12, sm: 6 }}>
              <FormControl fullWidth size="small" sx={{ mb: 2 }}>
                <InputLabel>Payment Type</InputLabel>
                <Select
                  value={paymentType}
                  onChange={(e) => setPaymentType(e.target.value)}
                  label="Payment Type"
                >
                  {paymentTypes.map((type) => (
                    <MenuItem key={type} value={type}>
                      {type}
                    </MenuItem>
                  ))}
                </Select>
              </FormControl>
            </Grid>

            <Grid size={{ xs: 12 }}>
              <TextField
                label="Meter Number"
                fullWidth
                size="small"
                value={meterNumber}
                onChange={(e) => setMeterNumber(e.target.value)}
                sx={{ mb: 2 }}
              />
            </Grid>

            <Grid size={{ xs: 12 }}>
              <Button
                variant="contained"
                color="primary"
                fullWidth
                onClick={handleVerify}
                disabled={!provider || !paymentType || !meterNumber}
                sx={{ py: 1.5, mb: 2 }}
              >
                Verify Meter
              </Button>
            </Grid>
          </Grid>
        </CardContent>
      </Card>

      {isVerified && (
        <Card elevation={3}>
          <CardContent>
            <Typography variant="h6" gutterBottom sx={{ fontWeight: "bold" }}>
              Customer Details
            </Typography>
            <Divider sx={{ mb: 3 }} />

            <Paper elevation={0} sx={{ p: 2, mb: 3, bgcolor: "grey.100" }}>
              <Grid container spacing={2}>
                <Grid size={{ xs: 12, sm: 6 }}>
                  <Typography variant="body1">
                    <strong>Name:</strong> {customerName}
                  </Typography>
                </Grid>
                <Grid size={{ xs: 12, sm: 6 }}>
                  <Typography variant="body1">
                    <strong>Address:</strong> {customerAddress}
                  </Typography>
                </Grid>
                <Grid size={{ xs: 12, sm: 6 }}>
                  <Typography variant="body1">
                    <strong>Meter Type:</strong> {meterType}
                  </Typography>
                </Grid>
                <Grid size={{ xs: 12, sm: 6 }}>
                  <Typography variant="body1">
                    <strong>Wallet Balance:</strong> ₦{walletBalance}
                  </Typography>
                </Grid>
              </Grid>
            </Paper>

            <Grid container spacing={2}>
              <Grid size={{ xs: 12, sm: 6 }}>
                <TextField
                  label="Amount (₦)"
                  fullWidth
                  size="small"
                  type="number"
                  value={amount}
                  onChange={(e) => {
                    const input = e.target.value;
                    if (parseFloat(input) >= 0 || input === "") {
                      setAmount(input);
                    }
                  }}
                  sx={{ mb: 2 }}
                />
              </Grid>
              <Grid size={{ xs: 12, sm: 6 }}>
                <TextField
                  label="Customer Number"
                  fullWidth
                  size="small"
                  value={customerNumber}
                  onChange={(e) => setCustomerNumber(e.target.value)}
                  sx={{ mb: 2 }}
                />
              </Grid>
              <Grid size={{ xs: 12 }}>
                <Button
                  variant="contained"
                  color="primary"
                  fullWidth
                  onClick={handleSubmit}
                  disabled={!amount || !customerNumber}
                  sx={{ py: 1.5 }}
                >
                  Pay Electricity Bill
                </Button>
              </Grid>
            </Grid>
          </CardContent>
        </Card>
      )}

      {/* Confirmation Modal */}
      <ConfirmModal
        open={isConfirmModalOpen}
        onClose={() => setIsConfirmModalOpen(false)}
        onConfirm={handleConfirm}
        transactionDetails={{
          amount,
          recipient: customerName,
          date: new Date().toLocaleString(),
          message: transactionDetails.finalMessage,
        }}
      />

      {/* Auth Modal */}
      <AuthModal
        open={isAuthModalOpen}
        onClose={() => setIsAuthModalOpen(false)}
        onVerify={handleProcessPayment}
      />

      {/* Transaction Modal */}
      <TransactionModal
        open={isTransactionModalOpen}
        onClose={handleTransactionModalClose}
        transactionDetails={transactionDetails}
        isLoading={isLoading}
      />
    </Box>
  );
};

export default BillPayment;
