"use client";

import React, { useState, useEffect } from "react";
import {
  Typography,
  Grid,
  TextField,
  Button,
  ToggleButton,
  Box,
  MenuItem,
  Select,
  Radio,
  RadioGroup,
  FormControlLabel,
  FormControl,
  FormLabel,
  Snackbar,
  Alert,
  Backdrop,
  Card,
  CardContent,
  Divider,
  Paper,
  useMediaQuery,
  Theme,
  InputLabel,
} from "@mui/material";
import Image from "next/image";
import gotvLogo from "../../../public/images/GOTV.png";
import dstvLogo from "../../../public/images/DStv.png";
import startimeLogo from "../../../public/images/startimes.jpg";
import showmaxLogo from "../../../public/images/Showmax.png";
import { cablePlans, fetchCablePlans } from "@/utils/cableplans";
import { AuthModal, ConfirmModal } from "@/app/components/modals/authmodal";
import TransactionModal from "@/app/components/modals/TransactionModal";
import axios from "axios";
import { useTheme } from "@mui/material/styles";
import { useSession } from "next-auth/react";
import LoadingComponent from "@/app/components/loadingcomponent";

const cableProviders = ["GOTV", "DSTV", "Startimes", "Showmax"];

type CableProvider = keyof typeof cablePlans;

const CablePayment: React.FC = () => {
  const { data: session } = useSession();
  const [selectedCableProvider, setSelectedCableProvider] = useState<
    CableProvider | ""
  >("");
  const [selectedCablePlan, setSelectedCablePlan] = useState<string | null>(
    null
  );
  const [iucNumber, setIucNumber] = useState<string>("");
  const [customerNumber, setCustomerNumber] = useState<string>("");
  const [isVerifying, setIsVerifying] = useState(false);
  const [amount, setAmount] = useState("");
  const [verificationResult, setVerificationResult] = useState<{
    Customer_Name: string;
    Current_Bouquet: string;
    Renewal_Amount: number;
  } | null>(null);
  const [isLoading, setIsLoading] = useState(false);
  const [isConfirmModalOpen, setIsConfirmModalOpen] = useState(false);
  const [isAuthModalOpen, setIsAuthModalOpen] = useState(false);
  const [isTransactionModalOpen, setIsTransactionModalOpen] = useState(false);
  const [transactionDetails, setTransactionDetails] = useState({
    finalMessage: "",
    amount: "",
    recipient: "",
    date: "",
    status: "success" as "success" | "pending" | "failed",
  });
  const [action, setAction] = useState<"renew" | "change" | null>(null);
  const [isBackdropOpen, setIsBackdropOpen] = useState(false);
  const [planName, setPlanName] = useState("");
  const [transactionPinStatus, setTransactionPinStatus] =
    useState<boolean>(false);
  const [thumbPrintStatus, setThumbPrintStatus] = useState<boolean>(false);
  const [snackbarOpen, setSnackbarOpen] = useState(false);
  const [snackbarMessage, setSnackbarMessage] = useState("");
  const [walletBalance, setWalletBalance] = useState<string>("");

  const theme = useTheme();
  const isMobile = useMediaQuery(theme.breakpoints.down("sm"));

  useEffect(() => {
    if (selectedCableProvider) {
      fetchCablePlans(selectedCableProvider.toLocaleLowerCase());
    }
  }, [selectedCableProvider]);

  useEffect(() => {
    const fetchUserType = async () => {
      try {
        const response = await axios.get(`/api/users/getSingleUser`);
        setWalletBalance(response.data.accountBalance);
        setTransactionPinStatus(response.data.transactionStatus);
        setThumbPrintStatus(response.data.thumbprintStatus);
      } catch (error) {
        console.error("Error fetching user type:", error);
      }
    };

    fetchUserType();
  }, [session]);

  const handleVerifyIUC = async () => {
    if (!selectedCableProvider || !iucNumber) {
      setSnackbarMessage("Please fill in all required fields.");
      setSnackbarOpen(true);
      return;
    }

    setIsVerifying(true);
    setIsBackdropOpen(true);
    try {
      const response = await axios.post("/api/cablebillverification", {
        provider: selectedCableProvider,
        iucNumber,
      });

      if (response.data.code === "000") {
        if (response.data.content.error) {
          setVerificationResult(null);
          setSnackbarMessage(
            `Verification Failed: ${response.data.content.error}`
          );
        } else {
          setSnackbarMessage("Verification successful.");
          setVerificationResult(response.data.content);
        }
      } else {
        setVerificationResult(null);
        setSnackbarMessage("Unable to confirm IUC number.");
      }
    } catch (error) {
      console.error("Error verifying IUC number:", error);
      setSnackbarMessage("Error verifying IUC number. Please try again.");
    } finally {
      setIsVerifying(false);
      setIsBackdropOpen(false);
      setSnackbarOpen(true);
    }
  };

  const handlePlanSelection = (variationCode: string) => {
    const selectedPlan = cablePlans[selectedCableProvider]?.find(
      (plan) => plan.variation_code === variationCode
    );
    if (selectedPlan) {
      setPlanName(selectedPlan.name);
      setSelectedCablePlan(variationCode);
      setAmount(selectedPlan.variation_amount);
    }
  };

  const handleSubmit = () => {
    if (
      !selectedCableProvider ||
      !selectedCablePlan ||
      !iucNumber ||
      !customerNumber
    ) {
      setSnackbarMessage("Please fill in all required fields.");
      setSnackbarOpen(true);
      return;
    }

    if (parseFloat(walletBalance) >= parseFloat(amount)) {
      setTransactionDetails({
        finalMessage: `Are you sure you want to ${
          action === "renew" ? "renew" : "change"
        } ${
          verificationResult?.Customer_Name || "Customer"
        }'s ${selectedCableProvider}'s subscription using ${planName}?`,
        amount,
        recipient: verificationResult?.Customer_Name || "Customer",
        date: new Date().toLocaleString(),
        status: "success",
      });

      setIsConfirmModalOpen(true);
    } else {
      setSnackbarMessage("Insufficient Balance, Please Fund your Wallet");
      setSnackbarOpen(true);
      setIsConfirmModalOpen(false);
    }
  };

  const handleConfirm = () => {
    setIsConfirmModalOpen(false);
    if (thumbPrintStatus || transactionPinStatus) {
      setIsAuthModalOpen(true);
    } else {
      handleVerify();
    }
  };

  const handleVerify = async (pin?: string) => {
    setIsAuthModalOpen(false);
    setIsLoading(true);
    setIsBackdropOpen(true);

    try {
      const response = await fetch("/api/buyCable", {
        method: "POST",
        headers: { "Content-Type": "application/json" },
        body: JSON.stringify({
          provider: selectedCableProvider,
          iucNumber,
          cablePlan: selectedCablePlan,
          amount,
          customerNumber,
          action: action,
        }),
      });

      const result = await response.json();

      setTransactionDetails((prev) => ({
        ...prev,
        finalMessage:
          result?.code === "000" &&
          result?.content?.transactions?.status === "delivered"
            ? `Payment of ₦${amount} for ${selectedCableProvider}'s ${selectedCablePlan} subscription for ${iucNumber} was successful.`
            : result?.code === "000" &&
              result?.content?.transactions?.status === "pending"
            ? `Payment of ₦${amount} for ${selectedCableProvider}'s ${selectedCablePlan} subscription for ${iucNumber} is pending. Please check back later.`
            : result?.code === "016" &&
              result?.content?.transactions?.status === "failed"
            ? `Payment failed due to an issue with the transaction. Please try again.`
            : result?.code === "018" ||
              result?.content?.transactions?.status === "failed"
            ? `Payment failed. Please try again.`
            : `Payment status unknown. Please contact support.`,
        status:
          result?.code === "000" &&
          result?.content?.transactions?.status === "delivered"
            ? "success"
            : result?.code === "000" &&
              result?.content?.transactions?.status === "pending"
            ? "pending"
            : result?.code === "016" &&
              result?.content?.transactions?.status === "failed"
            ? "failed"
            : result?.code === "018" ||
              result?.content?.transactions?.status === "failed"
            ? "failed"
            : "failed",
      }));

      setSnackbarMessage(
        result?.code === "000" &&
          result?.content?.transactions?.status === "delivered"
          ? "Payment successful."
          : "Payment failed."
      );
    } catch (error) {
      setTransactionDetails((prev) => ({
        ...prev,
        finalMessage: "Payment processing error. Please try again.",
        status: "failed",
      }));
      setSnackbarMessage("Payment processing error. Try again.");
    } finally {
      setIsTransactionModalOpen(true);
      setIsLoading(false);
      setIsBackdropOpen(false);
      setSnackbarOpen(true);
    }
  };

  const handleTransactionModalClose = () => {
    setIsTransactionModalOpen(false);
    window.location.reload();
  };

  return (
    <Box sx={{ maxWidth: 800, margin: "auto", p: isMobile ? 2 : 3 }}>
      {/* Backdrop for Loading */}
      <Backdrop
        open={isBackdropOpen}
        sx={{ color: "#fff", zIndex: (theme) => theme.zIndex.drawer + 1 }}
      >
        <LoadingComponent />
      </Backdrop>

      {/* Snackbar for Notifications */}
      <Snackbar
        open={snackbarOpen}
        autoHideDuration={3000}
        onClose={() => setSnackbarOpen(false)}
        anchorOrigin={{ vertical: "top", horizontal: "center" }}
      >
        <Alert
          severity={
            transactionDetails.status === "success" ? "success" : "error"
          }
          sx={{ width: "100%" }}
        >
          {snackbarMessage}
        </Alert>
      </Snackbar>

      <Card elevation={3} sx={{ mb: 3 }}>
        <CardContent>
          <Typography variant="h6" gutterBottom sx={{ fontWeight: "bold" }}>
            Cable TV Subscription
          </Typography>
          <Divider sx={{ mb: 3 }} />

          <Typography
            variant="subtitle1"
            gutterBottom
            sx={{ fontWeight: "bold" }}
          >
            Select Provider
          </Typography>
          <Grid container spacing={2} sx={{ mb: 3 }}>
            {cableProviders.map((provider) => (
              <Grid size={{ xs: 6, sm: 3 }} key={provider}>
                <ToggleButton
                  value={provider}
                  selected={selectedCableProvider === provider}
                  onChange={() =>
                    setSelectedCableProvider(provider as CableProvider)
                  }
                  sx={{
                    width: "100%",
                    height: isMobile ? "80px" : "120px",
                    p: 1,
                    flexDirection: "column",
                  }}
                >
                  <Box
                    sx={{
                      position: "relative",
                      width: "100%",
                      height: "60%",
                      mb: 1,
                    }}
                  >
                    <Image
                      src={
                        provider === "GOTV"
                          ? gotvLogo
                          : provider === "DSTV"
                          ? dstvLogo
                          : provider === "Startimes"
                          ? startimeLogo
                          : showmaxLogo
                      }
                      alt={provider}
                      fill
                      sizes="100%"
                      priority
                      style={{ objectFit: "contain" }}
                    />
                  </Box>
                  <Typography variant="caption">{provider}</Typography>
                </ToggleButton>
              </Grid>
            ))}
          </Grid>

          {selectedCableProvider && (
            <>
              <Typography
                variant="subtitle1"
                gutterBottom
                sx={{ fontWeight: "bold" }}
              >
                IUC Number
              </Typography>
              <Grid container spacing={2} alignItems="center">
                <Grid size={{ xs: 12, sm: 8 }}>
                  <TextField
                    fullWidth
                    size="small"
                    placeholder="Enter IUC number"
                    value={iucNumber}
                    onChange={(e) => setIucNumber(e.target.value)}
                  />
                </Grid>
                <Grid size={{ xs: 12, sm: 4 }}>
                  <Button
                    variant="contained"
                    fullWidth
                    onClick={handleVerifyIUC}
                    disabled={!iucNumber || isVerifying}
                    sx={{ height: "40px" }}
                  >
                    {isVerifying ? <LoadingComponent /> : "Verify IUC"}
                  </Button>
                </Grid>
              </Grid>
            </>
          )}
        </CardContent>
      </Card>

      {verificationResult && (
        <Card elevation={3}>
          <CardContent>
            <Typography variant="h6" gutterBottom sx={{ fontWeight: "bold" }}>
              Customer Details
            </Typography>
            <Divider sx={{ mb: 3 }} />

            <Paper elevation={0} sx={{ p: 2, mb: 3, bgcolor: "grey.100" }}>
              <Grid container spacing={2}>
                <Grid size={{ xs: 12, sm: 6 }}>
                  <Typography variant="body1">
                    <strong>Name:</strong> {verificationResult.Customer_Name}
                  </Typography>
                </Grid>
                <Grid size={{ xs: 12, sm: 6 }}>
                  <Typography variant="body1">
                    <strong>Current Plan:</strong>{" "}
                    {verificationResult.Current_Bouquet}
                  </Typography>
                </Grid>
                <Grid size={{ xs: 12, sm: 6 }}>
                  <Typography variant="body1">
                    <strong>Renewal Amount:</strong> ₦
                    {verificationResult.Renewal_Amount}
                  </Typography>
                </Grid>
                <Grid size={{ xs: 12, sm: 6 }}>
                  <Typography variant="body1">
                    <strong>Wallet Balance:</strong> ₦{walletBalance}
                  </Typography>
                </Grid>
              </Grid>
            </Paper>

            {(selectedCableProvider === "DSTV" ||
              selectedCableProvider === "GOTV") && (
              <>
                <Typography
                  variant="subtitle1"
                  gutterBottom
                  sx={{ fontWeight: "bold" }}
                >
                  Select Action
                </Typography>
                <FormControl component="fieldset" sx={{ mb: 3 }}>
                  <RadioGroup
                    row
                    value={action}
                    onChange={(e) =>
                      setAction(e.target.value as "renew" | "change")
                    }
                  >
                    <FormControlLabel
                      value="renew"
                      control={<Radio />}
                      label="Renew Plan"
                    />
                    <FormControlLabel
                      value="change"
                      control={<Radio />}
                      label="Change Plan"
                    />
                  </RadioGroup>
                </FormControl>
              </>
            )}

            <Typography
              variant="subtitle1"
              gutterBottom
              sx={{ fontWeight: "bold" }}
            >
              Select Plan
            </Typography>
            <FormControl fullWidth size="small" sx={{ mb: 3 }}>
              <InputLabel>Choose a plan</InputLabel>
              <Select
                value={selectedCablePlan || ""}
                onChange={(e) => handlePlanSelection(e.target.value)}
                label="Choose a plan"
              >
                <MenuItem value="" disabled>
                  Select Plan
                </MenuItem>
                {(cablePlans[selectedCableProvider] || []).map((plan) => (
                  <MenuItem
                    key={plan.variation_code}
                    value={plan.variation_code}
                  >
                    {plan.name} - ₦{plan.variation_amount}
                  </MenuItem>
                ))}
              </Select>
            </FormControl>

            <Typography
              variant="subtitle1"
              gutterBottom
              sx={{ fontWeight: "bold" }}
            >
              Customer Number
            </Typography>
            <TextField
              fullWidth
              size="small"
              placeholder="Enter Customer Number"
              value={customerNumber}
              onChange={(e) => setCustomerNumber(e.target.value)}
              sx={{ mb: 3 }}
            />

            <Button
              variant="contained"
              color="primary"
              fullWidth
              onClick={handleSubmit}
              disabled={!selectedCablePlan || !customerNumber}
              size="large"
            >
              {action === "renew"
                ? "Renew Subscription"
                : action === "change"
                ? "Change Plan"
                : "Subscribe"}
            </Button>
          </CardContent>
        </Card>
      )}

      {/* Confirmation Modal */}
      <ConfirmModal
        open={isConfirmModalOpen}
        onClose={() => setIsConfirmModalOpen(false)}
        onConfirm={handleConfirm}
        transactionDetails={{
          message: transactionDetails.finalMessage,
          amount,
          recipient: transactionDetails.recipient,
          date: transactionDetails.date,
        }}
      />

      {/* Auth Modal */}
      <AuthModal
        open={isAuthModalOpen}
        onClose={() => setIsAuthModalOpen(false)}
        onVerify={handleVerify}
      />

      {/* Transaction Modal */}
      <TransactionModal
        open={isTransactionModalOpen}
        onClose={handleTransactionModalClose}
        transactionDetails={transactionDetails}
        isLoading={isLoading}
      />
    </Box>
  );
};

export default CablePayment;
