import Google from "next-auth/providers/google";
import EmailProvider from "next-auth/providers/email";
import client from "./lib/db";
import { MongoDBAdapter } from "@auth/mongodb-adapter";
import { clearStaleTokens } from "./lib/clearStaleTokensServerAction";
import { NextAuthOptions } from "next-auth";
import dbConnect from "./lib/connectdb";
import { User } from "./models/user";
import { createTransport } from "nodemailer";
import { generateReferralCode } from "./utils/junkfunctions";

export const authOptions = {
  adapter: MongoDBAdapter(client),
  secret: process.env.AUTH_SECRET as string,

  session: {
    strategy: "jwt",
    maxAge: 30 * 1 * 60 * 60,
  },
  pages: {
    signIn: "/auth/sign-in",
    verifyRequest: "/auth/auth-success",
    error: "/auth/auth-error",
  },
  providers: [
    Google({
      clientId: process.env.AUTH_GOOGLE_ID as string,
      clientSecret: process.env.AUTH_GOOGLE_SECRET as string,
      allowDangerousEmailAccountLinking: true,
      authorization: {
        params: {
          prompt: "select_account",
          access_type: "offline",
          response_type: "code",
        },
      },
      async profile(profile) {
        const blockedEmails = [
          "wintosam@gmail.com",
          "williamwoodland33@gmail.com",
          "jochebedapps@gmail.com",
          "princeichulo@gmail.com",
        ];

        const email = profile.email ?? "";

        const isProsperSeries =
          /^prosperjasper\d{3}@gmail\.com$/.test(email) ||
          email === "prosperjasper@gmail.com";

        if (blockedEmails.includes(email) || isProsperSeries) {
          throw new Error("Blocked email address");
        }
        return {
          id: profile.sub,
          username: profile.sub,
          email: profile.email,
          emailVerified: profile.email_verified,
          name: profile.name,
          image: profile.picture,
          role: profile.role ?? "user",
          provider: profile.provider ?? "google",
          lastLogin: null,
          hasSeenModal: false,
          referralCode: await generateReferralCode(), // Generate a unique referral code
        };
      },
      httpOptions: {
        timeout: 20000,
      },
    }),
    EmailProvider({
      server: {
        host: process.env.EMAIL_SERVER_HOST as string,
        port: parseInt(process.env.EMAIL_SERVER_PORT as string, 10),
        auth: {
          user: process.env.EMAIL_SERVER_USER as string,
          pass: process.env.EMAIL_SERVER_PASSWORD as string,
        },
      },
      from: process.env.EMAIL_FROM as string,
      sendVerificationRequest: async ({ identifier, url, provider }) => {
        await dbConnect();

        // Check if the user already exists
        let user = await User.findOne({ email: identifier });

        if (!user) {
          // Create a new user if they don't exist
          user = await User.create({
            email: identifier,
            referralCode: generateReferralCode(), // Generate a unique referral code
          });
        }
        const { host } = new URL(url);
        const transport = createTransport(provider.server);
        await transport.sendMail({
          to: identifier,
          from: provider.from,
          subject: `Sign in to ${host}`,
          text: `Sign in by clicking on the link below:\n\n${url}\n\n`,
          html: `<p>Sign in by clicking on the link below:</p><p><a href="${url}">Sign in</a></p>`,
        });
      },
    }),
  ],
  callbacks: {
    async signIn({ user }) {
      const blockedEmails = [
        "wintosam@gmail.com",
        "williamwoodland33@gmail.com",
        "jochebedapps@gmail.com",
        "princeichulo@gmail.com",
      ];

      const email = user.email ?? "";

      const isProsperSeries =
        /^prosperjasper\d{3}@gmail\.com$/.test(email) ||
        email === "prosperjasper@gmail.com";

      if (blockedEmails.includes(email) || isProsperSeries) {
        console.warn(`Blocked login attempt from: ${email}`);
        return false; // Block sign-in
      }
      return true; // Allow sign-in
    },
    async redirect({ url, baseUrl }) {
      return baseUrl + "/dashboard";
    },
    async jwt({ token, trigger, session, user }) {
      if (user) {
        token.email = user.email;
        token.name = user.name;
        token.id = user.id;
        token.image = user.image;
        token.role = user.role;
        token.hasSeenModal = user.hasSeenModal;
        token.lastLogin = user.lastLogin;
        await clearStaleTokens();
      }
      return token;
    },
    async session({ session, token }) {
      await dbConnect();
      const user = await User.findOne({ email: session?.user?.email });

      if (user) {
        user.lastLogin = new Date();
        await user.save();
        session.user = {
          id: user._id.toString(),
          name: user.name,
          email: user.email,
          image: user.image,
          role: user.role,
          hasSeenModal: user.hasSeenModal,
          lastLogin: user.lastLogin,
          provider: token.provider,
          // any other SAFE fields you want
        };
        // session.user = { ...session.user, ...user.toObject() };
      } else {
        session.user = {
          hasSeenModal: token.hasSeenModal,
          email: token.email,
          name: token.name,
          id: token.id,
          image: token.image,
          role: token.role,
          lastLogin: token.lastLogin,
          provider: token.provider,
        };
      }
      return session;
    },
  },
  debug: true,
} as NextAuthOptions;
