import { NextResponse } from "next/server";
import type { NextRequest } from "next/server";
import { getToken } from "next-auth/jwt";

const secret = process.env.NEXTAUTH_SECRET;

// Expanded list of public paths from sitemap
const publicPaths = [
  "/",
  "/manifest.webmanifest",
  "/auth/sign-in",
  "/auth/auth-success",
  "/auth/auth-error",
  "/contact-sabamu-enterprice",
  "/about-sabamuent-vtu-business",
  "/vtu-services-in-nigeria",
  "/vtu-pricing",
  "/terms",
  "/privacy",
];

export async function middleware(req: NextRequest) {
  const { pathname } = req.nextUrl;

  // 1. Allow Next.js internals and API routes
  if (pathname.startsWith("/_next") || pathname.startsWith("/api")) {
    return NextResponse.next();
  }

  // 2. Allow all static public paths
  if (publicPaths.includes(pathname)) {
    return NextResponse.next();
  }

  // 3. Get token for authentication
  const token = await getToken({ req, secret });

  // 4. Redirect unauthenticated access to protected pages
  if (!token) {
    if (
      pathname.startsWith("/dashboard") ||
      pathname.startsWith("/admindashboard")
    ) {
      const signInUrl = req.nextUrl.clone();
      signInUrl.pathname = "/auth/sign-in";
      return NextResponse.redirect(signInUrl);
    }
    return NextResponse.next();
  }

  // 5. Role-based access control for admin
  const role = token?.role;
  if (role !== "admin" && pathname.startsWith("/admindashboard")) {
    const dashboardUrl = req.nextUrl.clone();
    dashboardUrl.pathname = "/dashboard";
    return NextResponse.redirect(dashboardUrl);
  }

  return NextResponse.next();
}

export const config = {
  matcher: [
    "/((?!_next|api).*)", // Middleware runs on all paths except `_next` and `api`
  ],
};
