import mongoose, { Schema, Document, model } from "mongoose";

export interface ITransaction extends Document {
  type: string; // "CableSubscription", "BillPayment", "ManualFunding", "AutomaticFunding", "DataTransaction", "AirtimeTransaction"
  amount?: number; // General transaction amount
  balanceBefore: number; // Balance before transaction
  balanceAfter: number; // Balance after transaction
  status: string; // "Pending", "Successful", "Failed"
  adminEmail: string; // Email of the admin who processed the transaction
  userEmail: string;
  userId?: string;
  bypass: boolean;
  referenceId: string;

  // Airtime Transaction Fields
  airtimeType: string;
  buyingPrice: number;
  // Cable Subscription Fields
  iucOrSmartcardNumber?: number;
  provider: string;
  cablePlan: string;
  action: string;
  refund?: boolean;

  // Bill Payment Fields
  meterNumber?: number;
  token?: string;
  customerPhone?: number;
  meterType?: string;
  request_id?: string;
  customerName?: string;
  customerAddress?: string;
  discoName?: string;

  // Data Transaction Fields
  planType?: string;
  plansize: string;
  mobileNumber?: number;
  medium?: string; // e.g., "API", "Web"
  network?: string;

  // Funding Transaction Fields
  fundingType?: "Manual" | "Automatic"; // Specific to funding transactions
  fundingSource?: "API" | "Admin" | "Referral" | "ONE TIME PAYMENT"; // Specific to funding transactions
}

const transactionSchema = new Schema<ITransaction>(
  {
    type: {
      type: String,
      required: true,
      enum: [
        "CableSubscription",
        "BillPayment",
        "ManualFunding",
        "AutomaticFunding",
        "DataTransaction",
        "AirtimeTransaction",
        "WithdrawalTransaction",
        "ReferalBonus",
        "RefundTransaction",
      ],
    },
    amount: { type: Number },
    balanceBefore: { type: Number },
    balanceAfter: { type: Number },
    adminEmail: { type: String },
    status: {
      type: String,
      required: true,
      enum: ["Pending", "Successful", "Failed"],
    },
    referenceId: { type: String, unique: true },

    // Airtime Subscription Field
    airtimeType: { type: String },
    buyingPrice: { type: Number },

    // Cable Subscription Fields
    iucOrSmartcardNumber: { type: Number },
    request_id: { type: String },
    provider: { type: String },
    cablePlan: { type: String },
    action: { type: String },

    userId: { type: String },
    userEmail: { type: String },
    refund: { type: Boolean, default: false },
    bypass: { type: Boolean, default: false },

    // Bill Payment Fields
    meterNumber: { type: Number },
    token: { type: String },
    customerPhone: { type: Number },
    meterType: { type: String },
    customerName: { type: String },
    customerAddress: { type: String },
    discoName: { type: String },

    // Data Transaction Fields
    planType: { type: String },
    mobileNumber: { type: Number },
    medium: { type: String },
    network: { type: String },
    plansize: { type: String },

    // Funding Transaction Fields
    fundingType: { type: String, enum: ["Manual", "Automatic"] },

    fundingSource: {
      type: String,
      enum: ["API", "Admin", "Referral", "ONE TIME PAYMENT"],
    },
  },
  {
    timestamps: true, // Automatically manage createdAt and updatedAt fields
  }
);

// Create the model if it doesn't already exist
const Transaction =
  mongoose.models.Transaction ||
  model<ITransaction>("Transaction", transactionSchema);

export default Transaction;
